/* Scythe_Optimize.h */
//
// This library is the declaration of the Optimizing functions
// for the SCYTHE project.  
//
// Scythe C++ Library
// Copyright (C) 2000 Kevin M. Quinn and Andrew D. Martin
//
// This code written by:
//
// Kevin Quinn
// Assistant Professor
// Dept. of Political Science and 
// Center for Statistics and the Social Sciences
// Box 354322
// University of Washington
// Seattle, WA  98195-4322
// quinn@stat.washington.edu
//
// Andrew D. Martin
// Assistant Professor
// Dept. of Political Science
// Campus Box 1063
// Washington University
// St. Louis, MO 63130
// admartin@artsci.wustl.edu
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA

#ifndef SCYTHE_OPTIMIZE_H
#define SCYTHE_OPTIMIZE_H

#include <fstream.h>
#include <iomanip.h>
#include <math.h>
#include "Scythe_Double_Matrix.h"
#include "Scythe_Simulate.h"

// Avoid NameSpace Pollution
namespace SCYTHE {
  
  
  
  /**********************************************************************/
  //  FUNCTION: Intsimp - Calculates the Definite integral of a function
  //      from a to b
  //  Input:  fun -  pointer to function to integrate. fun must take a
  //                 double argument and return a double
  //          a   - lower limit of integration
  //          b   - upper limit of integration
  //          N   - number of subintervals
  //  Return:  Double I    approximate value of integral
  //  Usage: intsimp(double (*fun)(const double&), const double& a,
  //                      const double& b, const int N);
  //  Errors:  2001-2002
  //  Dependencies:  none
  
  double intsimp(double (*fun)(const double&), const double& a,
		 const double& b, const int N);
  
  
  /**********************************************************************/
  //  FUNCTION: Adaptsimp - Calculates the Definite Integral from a to b
  //        of a given function
  //  Input:  fun - pointer to function to integrate. fun must take a
  //                 double argument and return a double
  //          a   - lower limit of integration
  //          b   - upper limit of integration
  //          N   - number of subintervals
  //          tol - tolerance
  //  Return:  Double  I - Approximate value of integral
  //  Usage: adaptsimp(double (*fun)(double&), double& a,
  //                    double& b,  int N, double tol=1e-5);
  //  Errors:  2003-2004
  //  Dependencies:  none
  
  double adaptsimp(double (*fun)(const double&), const double& a,
		   const double& b, const int N, 
		   const double tol=1e-5);
  
  
  /**********************************************************************/
  //  FUNCTION: Gradfdif - Numerically calculates the gradient of a 
  //        function at theta using a forward difference formula
  //  Input:  fun   - pointer to function to differentiate. fun must take
  //                  3 Matrix arguments and return a double
  //          theta - Matrix of parameter values at which to calculate
  //                  the gradient theta assumed to be a column vector
  //          y     - Matrix argument that is sent to fun
  //          X     - Matrix argument that is sent to fun
  //  Return:  Matrix object grad - Gradient vector at theta
  //  Usage: gradfdif(double (*fun)(Matrix&), Matrix& theta);
  //  Errors:  2005
  //  Dependencies:  none
  
  Matrix gradfdif(double (*fun)(const Matrix&, const Matrix&, const Matrix&), 
		  const Matrix& theta, const Matrix& y, const Matrix& X);
  
  
  /**********************************************************************/
  //  FUNCTION: Gradfdifls - Numerically calculates the first deriv. of
  //        a function wrt alpha at (theta + alpha*p) using a
  //        forward difference formula. Primarily useful in linesearches.
  //  Input:  fun   - pointer to function to differentiate. fun must take 3 
  //                  Matrix arguments and return a double
  //          alpha - double step length
  //          theta - Matrix of parameter values at which to calculate the 
  //                  gradient. Theta is assumed to be a column vector
  //          p     - direction vector
  //          y     - Matrix argument sent to fun
  //          X     - Matrix argument sent to fun
  //  Return:  Double deriv - First Derivative of function
  //  Usage: gradfdifls(double (*fun)(Matrix&), double& alpha,
  //                       Matrix& theta, Matrix& p);
  //  Errors: 2006-2007
  //  Dependencies:  none
  
  double gradfdifls(double (*fun)(const Matrix&, const Matrix&, 
				  const Matrix&), const double& alpha,
		    const Matrix& theta, const Matrix& p, const Matrix& y,
		    const Matrix& X);
  
  
  /**********************************************************************/
  //  FUNCTION: Jacfdif - Numerically calculates the Jacobian of a
  //          function at theta using a forward difference formula
  //  Input:  fun   - pointer to function to differentiate. fun must take a 
  //                  Matrix argument and return a Matrix
  //          theta - Matrix of parameter values at which to calculate the 
  //                  Jacobian. Theta assumed to be a column vector
  //  Return:  Matrix object - Jacobian Matrix at theta
  //  Usage: jacfdif(Matrix (*fun)(Matrix&), Matrix& theta);
  //  Errors: 2008
  //  Dependencies:  none
  
  Matrix jacfdif(Matrix (*fun)(const Matrix&), const Matrix& theta);
  
  
  /**********************************************************************/
  //  FUNCTION: hesscdif - Numerically calculates the Hessian of a
  //          function at theta using a central difference formula
  //  Input:  fun   - pointer to function to differentiate. fun must take 3 
  //                  Matrix arguments and return a double
  //          theta - Matrix of parameter values at which to calculate the 
  //                  Hessian. Theta assumed to be a column vector
  //          y     - Matrix argument sent to fun
  //          X     - Matrix argument sent to fun
  //  Return:  Matrix object - Hessian Matrix at theta
  //  Usage: hesscdif(double (*fun)(Matrix&), Matrix& theta);
  //  Errors: XXXX
  //  Dependencies:  none
  
  Matrix hesscdif(double (*fun)(const Matrix&, const Matrix&, const Matrix&),
		  const Matrix& theta,
		  const Matrix& y, const Matrix& X);
  
  
  
  /**********************************************************************/
  //  FUNCTION: Linesearch1 - Performsa line search to find the step
  //        length alpha that approximately minimizes an implied 1d function
  //  Input: fun   - pointer to function to minimize. fun must take 3 Matrix
  //                 arguments and return a double
  //         theta - Matrix of parameter values that anchor the 1d function
  //         p     - direction vector that creates the 1d function
  //         y     - Matrix argument sent to fun
  //         X     - Matrix argument sent to fun
  //  Return:  Double giving the step length that approximately minimizes
  //           the 1d function (alpha)
  //  Usage: linesearch1(double (*fun)(Matrix&), Matrix& theta, Matrix& p);
  //  Errors: 2009-2010
  //  Dependencies:  none
  
  double linesearch1(double (*fun)(const Matrix&, const Matrix&, 
				   const Matrix&), const Matrix& theta,
		     const Matrix& p, const Matrix& y, const Matrix& X);
  
  
  /**********************************************************************/
  //  FUNCTION: Linesearch2 - Performsa line search to find the step
  //        length alpha that approximately minimizes an implied 1d function
  //  Input: fun   - pointer to function to minimize. fun must take 3 Matrix
  //                 arguments and return a double
  //         theta - Matrix of parameter values that anchor the 1d function
  //         p     - direction vector that creates the 1d function
  //         y     - Matrix argument sent to fun
  //         X     - Matrix argument sent to fun
  //  Return:  Double giving the step length that approximately minimizes   
  //           the 1d function (alpha)
  //  Usage: linesearch2(double (*fun)(Matrix&), Matrix& theta, Matrix& p);
  //  Errors: 2016-17
  //  Dependencies:  none
  
  double linesearch2(double (*fun)(const Matrix&, const Matrix&, 
				   const Matrix&), const Matrix& theta,
		     const Matrix& p, const Matrix& y, const Matrix& X);
  
  
  /**********************************************************************/
  //  FUNCTION: Zoom - Finds minimum of a function once bracketed
  //  Input:  fun   - pointer to function to minimize. fun must take 3 
  //                  Matrix arguments and return a double
  //          alo   - lower bracket
  //          ahi   - upper bracket
  //          theta - Matrix of parameter values that anchor the 1d function
  //          p     - direction vector that creates the 1d function
  //          y     - Matrix argument sent to fun
  //          X     - Matrix argument sent to fun
  //  Return:  Double giving the step length that approximately minimizes
  //           the 1d function (alpha)
  //  Usage: zoom(double (*fun)(Matrix&), double& alo, double& ahi, 
  //                  Matrix& theta, Matrix& p);
  //  Errors: 2011
  //  Dependencies:  none
  
  double zoom(double (*fun)(const Matrix&, const Matrix&, const Matrix&), 
	      const double& alo,
	      const double& ahi, const Matrix& theta, const Matrix& p,
	      const Matrix& y, const Matrix& X);
  
  
  /**********************************************************************/
  //  FUNCTION: BFGS - Numerically finds the minimum of a function fun
  //        using the BFGS algorithm
  //  Input:  fun   - pointer to function to minimize. fun must take 3 
  //                  Matrix arguments and return a double
  //          theta - Matrix of parameter values at which to start the 
  //                  optimization. Theta assumed to be a column vector
  //          y     - fixed variable sent to fun
  //          X     - fixed variable sent to fun
  //          maxit - maximum number of iterations
  //          tolerance - conergence tolerance
  //
  //  Return:  Thetamin - value of theta that minimizes the function
  //  Usage: BFGS(double (*fun)(Matrix&), Matrix& theta);
  //  Errors: 2013
  //  Dependencies:  none
  
  Matrix BFGS(double (*fun)(const Matrix&, const Matrix&, const Matrix&), 
	      const Matrix& theta, const Matrix& y = Matrix(1,1),
	      const Matrix& X = Matrix(1,1), const int& maxit=1000, 
	      const double& tolerance=1e-4);
  
  
  /**********************************************************************/
  // FUNCTION: Nls_broyden  - Solves a system of n nonlinear equations in
  //        n unknowns of the form:
  //
  //           fun(thetastar) = 0
  //
  //        for thetastar given starting value theta.  Uses Broyden's method.
  //  Input:  fun   - pointer to function to solve.  Function must take 
  //                  a Matrix argument and return a Matrix
  //          theta - Matrix of parameter values at which to start the
  //                  solve procedure. Theta assumed to be a column vector
  //          maxit - maximum number of iterations
  //          tolerance - conergence tolerance
  //  Return: Thetastar - The value of Theta that Zeros the function
  //  Usage: nls_broyden( Matrix (*fun)(Matrix&), Matrix& theta);
  //  Errors: 2013
  //  Dependencies:  none
  
  Matrix nls_broyden(Matrix (*fun)(const Matrix&), const Matrix& theta,
		     const int& maxit=5000, const double& tolerance=1e-6);
  
}  // namespace dec
#endif /* SCYTHE_OPTIMIZE_H */









