/* Scythe_Simulate.cc */
// Function definitions for probability density/distribution functions
// and pseudo randon number generators for the Scythe Statistical
// Library.
//
// Scythe C++ Library
// Copyright (C) 2000 Kevin M. Quinn and Andrew D. Martin
//
// This code written by:
//
// Kevin Quinn
// Assistant Professor
// Dept. of Political Science and 
// Center for Statistics and the Social Sciences
// Box 354322
// University of Washington
// Seattle, WA  98195-4322
// quinn@stat.washington.edu
//
// Andrew D. Martin
// Assistant Professor
// Dept. of Political Science
// Campus Box 1063
// Washington University
// St. Louis, MO 63130
// admartin@artsci.wustl.edu
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA

#ifndef SCYTHE_SIMULATE_CC
#define SCYTHE_SIMULATE_CC

#include "Scythe_Simulate.h"

// Avoid NameSpace Pollution
namespace SCYTHE {

// Initial values of random seeds for ranmars();
static unsigned long Z = 362436069;
static unsigned long W = 521288629;
static unsigned long JSR = 123456789;
static unsigned long JCONG = 380116160;

// Some more globals for ranmars()
static unsigned long X = 0, Y = 0, T[UCHAR_MAX + 1];
static unsigned char C = 0;

/**********************************************************************/
/*!
 * \brief Finds the minimum of two doubles
 *
 * This function finds the minimum of two doubles and returns a double.
 * \param a a constant reference to one of the variables being compared.
 * \param b a constant reference to one of the variables being compared.
 * \return a double, the minimum value of the two variables input.
 */
double
min (const double& a, const double& b)
{
  return (a>b) ? b : a;
}

/**********************************************************************/
/*!
 * \brief Finds the maximum of two doubles
 *
 * This function finds the maximum of two doubles and returns a double.
 * \param a a constant reference to one of the variables being compared.
 * \param b a constant reference to one of the variables being compared.
 * \return a double, the maximum value of the two variables input.
 */
double
max (const double& a, const double& b)
{
  return (a<b) ? b : a;
}

/**********************************************************************/
/*!
 * \brief Sets the random seed for \e ranmars()
 *
 * This function sets the random seed for the function \e ranmars().
 * \param z a constant reference to an unsigned long variable .
 * \param w a constant reference to an unsigned long variable.
 * \param jsr a constant reference to an unsigned long variable.
 * \param jcong a constant reference to an unsigned long variable.
 * \return void
*/
void setRandSeed(const unsigned long& z, const unsigned long& w,
		 const unsigned long& jsr, const unsigned long& jcong) {
  Z = z;
  W = w;
  JSR = jsr;
  JCONG = jcong;
}

/**********************************************************************/
/*!
 * \brief Generates a random uniform number
 *
 * This function generates a random uniform deviate from 0 to 1.
 * \param z a constant reference to an unsigned long variable .
 * \param w a constant reference to an unsigned long variable.
 * \param jsr a constant reference to an unsigned long variable.
 * \param jcong a constant reference to an unsigned long variable.
 * \return the random number (a double)
*/

double ranmars() {
  // Multiply With Carry Generator
  unsigned long mwc = ((Z=36969UL*(Z&65535UL)+(Z>>16))<<16) +
    (W=18000UL*(W&65535UL)+(W>>16));
  
  // Congruential Generator
  unsigned long cong = (JCONG=69069UL*JCONG+1234567UL);
  
  // 3-Shift-Register Generator
  JSR=JSR^(JSR<<17);
  JSR=JSR^(JSR>>13);
  unsigned long shr3 = (JSR=JSR^(JSR<<5));
  
  // Subtract With Borrow Generator
  X = T[ (unsigned char)(C+15)];
  T[(unsigned char)(C+237)] = X - (Y = T[(unsigned char)(C + 1)] +
				   (X < Y));
  unsigned long swb = T[++C];
  
  // Keep It Simple Stupid Generator
  unsigned long kiss = ((mwc ^ cong) + shr3);
  
  // We return a KISS+SWB (Should pass diehard and have period
  // > 2^7700).  We return a double in (0,1).
  
  return (kiss + swb) * 2.32830643708e-10;
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of random uniform numbers
 *
 * This function generates a Matrix of random uniform deviates from 
 * 0 to 1.
 * PLEASE NOTE: THIS FUNCTION IS OVERLOADED.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param columns a constant reference to the number of columns in the 
 * Matrix.
 * \return the Matrix of random numbers.
*/
Matrix
runif (const int& rows, const int& cols)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1001:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::runif(const int& rows, const int& cols)" 
         << endl;
    exit(1001);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = runif ();
  
  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a random normal number
 *
 * This function generates a random normal number according to the 
 * Box-Muller method.
 * \return the random number (a double).
*/
double
rnorm1 (void)
{
  static int rnorm_count = 1;
  static double x2;
  double nu1, nu2, rsquared, sqrt_term;
  if (rnorm_count == 1){// odd numbered passses
    do {
      nu1 = -1 +2*runif();
      nu2 = -1 +2*runif();
      rsquared = ::pow(nu1,2) + ::pow(nu2,2);
    }
    while (rsquared >= 1 || rsquared == 0.0);
    sqrt_term = ::sqrt(-2*::log(rsquared)/rsquared);
    x2 = nu2*sqrt_term;
    rnorm_count = 2;
    return nu1*sqrt_term;
  }
  else{// even numbered passes
    rnorm_count = 1;
    return x2;
  } 
}


/**********************************************************************/
/*!
 * \brief Generates a normal random number according to input parameters.
 *
 * This function generates a normal random numbers with input 
 * \e mu and \e sigma^2.
 * \param mu a constant reference to the mean (a double) for the 
 * random number.
 * \param sigma2 a constant reference to the variance (a double) 
 * for the random number.
 * \return the random number.
*/
double
rnorm (const double& mu, const double& sigma2)
{
  if (sigma2 <= 0) {
    cerr << "Error 1002:  negative variance in SCYTHE::"
	 << "rnorm(const double& mu, const double& sigma2)" << endl;
    exit (1002);
  } else
    return (mu + rnorm1 () * ::sqrt (sigma2));
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of normal random numbers according to input
 * parameters.
 *
 * This function generates a Matrix of normal random numbers 
 * with input \e mu and \e sigma^2.
 * \param rows a constant reference to the number of rows in 
 * the Matrix (an integer).
 * \param cols a constant reference to the number of rows in 
 * the Matrix (an integer).
 * \param mu a constant reference to the mean (a double) for 
 * the random number.
 * \param sigma2 a constant reference to the variance (a double) 
 * for the random number.
 * \return the random number.
*/
Matrix
rnorm (const int& rows, const int& cols, const double& mu,
       const double& sigma2)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1003:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rnorm(const int& rows, int& cols, "
         << "const double& mu, const double& sigma2)" 
         << endl;
    exit(1003);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rnorm (mu, sigma2);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a Gamma deviate according to input parameter
 *
 * This function generates a Gamma(\a alpha,1) deviate according 
 * to an input parameter.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Gamma distribution.
 * \return the random number.
 */
double
rgamma1 (const double& alpha)
{
  int test;
  double u, v, w, x, y, z, b, c;
  static double accept;

  // Check for allowable parameters
  if (alpha <= 1) {
    cerr << "Error 1004:  alpha < 1 in SCYTHE::" 
         << "rgamma1(const double& alpha)" << endl;
    exit (1004);
  }

  // Implement Best's (1978) simulator
  b = alpha - 1;
  c = 3 * alpha - 0.75;
  test = 0;
  while (test == 0) {
    u = runif ();
    v = runif ();
    
    w = u * (1 - u);
    y = ::sqrt (c / w) * (u - .5);
    x = b + y;

    if (x > 0) {
      z = 64 * ::pow (v, 2) * ::pow (w, 3);
      
      if (z <= (1 - (2 * ::pow (y, 2) / x))) {
        test = 1;
        accept = x;
      } else if ((2 * (b * ::log (x / b) - y)) >= ::log (z)) {
        test = 1;
        accept = x;
      } else {
        test = 0;
      }
    }
  }
  return (accept);
}

/**********************************************************************/
/*!
 * \brief Generates a Gamma deviate according to input parameter
 *
 * This function generates a Gamma(\a alpha,\a beta) deviate 
 * according to an input parameter.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Gamma distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Gamma distribution.
 * \return the random number.
*/
double
rgamma (const double& alpha, const double& beta)
{
  static double report;

  // Check for allowable parameters
  if (alpha <= 0) {
    cerr << "Error 1005:  alpha <= 0 in SCYTHE::" 
         << "rgamma(const double& alpha, const double& beta)" << endl;
    exit (1005);
  }
  if (beta <= 0) {
    cerr << "Error 1005:  beta <= 0 in SCYTHE::" 
         << "rgamma(const double& alpha, const double& beta)" << endl;
    exit (1006);
  }

  // alpha > 1
  if (alpha > 1)
    report = rgamma1 (alpha) / beta;

  // alpha == 1
  if (alpha == 1)
    report = -::log (runif ()) / beta;

  // alpha < 1
  if (alpha < 1)
    report = rgamma1 (alpha + 1) * ::pow (runif (), 1 / alpha)
             / beta;

  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Gamma deviates according to 
 * input parameter.
 *
 * This function generates a Matrix of Gamma(\a alpha,\a beta) 
 * deviates according to an input parameter.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Gamma distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Gamma distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rgamma (const int& rows, const int& cols, const double& alpha, 
        const double& beta)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1007:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rgamma(const int& rows, const int& cols,"
         << " const double& alpha, const double& beta)" << endl;
    exit(1007);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rgamma (alpha, beta);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a Inverse-Gamma deviate according 
 * to input parameters
 * This function generates a Inverse-Gamma(\a alpha,\a beta) 
 * deviate according to input parameters.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Inverse-Gamma distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Inverse-Gamma distribution.
 * \return the random number.
 */
double
rigamma (const double& alpha, const double& beta)
{
  static double report;

  // Check for allowable parameters
  if (alpha <= 0) {
    cerr << "Error 1008:  alpha <= 0 in SCYTHE::" 
         << "rigamma(const double& alpha, const double& beta)" << endl;
    exit (1008);
  }
  if (beta <= 0) {
    cerr << "Error 1009:  beta <= 0 in SCYTHE::" 
         << "rigamma(const double& alpha, const double& beta)" << endl;
    exit (1009);
  }
  // Return reciprocal of gamma deviate
  report = ::pow (rgamma (alpha, beta), -1);
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Inverse-Gamma deviates 
 * according to input parameters
 *
 * This function generates a Matrix of Inverse-Gamma(\a 
 * alpha,\a beta) deviates according to input parameters.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Inverse-Gamma distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Inverse-Gamma distribution.
 * \return the Matrix of random numbers.
*/
Matrix
rigamma (const int& rows, const int& cols, const double& alpha,
   const double& beta)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1010:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rigamma(const int rows&, const int&" 
         << " cols, const double& alpha, const double& beta)" << endl;
    exit (1010);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rigamma (alpha, beta);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a Chi-Squared deviate according to 
 * an input parameter
 *
 * This function generates a Chi-Squared(\a nu) deviate 
 * according to an input parameter.
 * \param nu a constant reference to the \a nu parameter 
 * from the Chi-Squared distribution.
 * \return the random number.
 */
double
rchisq (const double& nu)
{
  static double report;
  
  // Check for allowable paramter
  if (nu <= 0) {
    cerr << "Error 1011:  degrees of freedom <= 0 in "
         << "SCYTHE::rchisq(const double& nu)" << endl;
    exit (1011);
  }

  // Return Gamma(nu/2, 1/2) deviate
  report = rgamma (nu / 2, .5);
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Chi-Squared deviates 
 * according to an input parameter
 *
 * This function generates a Matrix of Chi-Squared(\a nu) 
 * deviates according to an input parameter.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param nu a constant reference to the \a nu parameter 
 * from the Chi-Squared distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rchisq (const int& rows, const int& cols, const double& nu)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1012:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rchisq(const int& rows, const int& " 
         << "cols, const double& nu)" << endl;
    exit (1012);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rchisq (nu);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates an Inverse-Chi-Squared deviate 
 * according to an input parameter
 *
 * This function generates an Inverse-Chi-Squared(\a nu) 
 * deviate according to an input parameter.
 * \param nu a constant reference to the \a nu parameter 
 * from the Inverse-Chi-Squared distribution.
 * \return the random number.
 */
double
richisq (const double& nu)
{
  static double report;
  
  // Check for allowable parameter
  if (nu <= 0) {
    cerr << "Error 1013:  degrees of freedom parameter <= 0 in " 
         << "SCYTHE::richisq(const double& nu)" << endl;
    exit (1013);
  }
  
  // Return Inverse-Gamma(nu/2, 1/2) deviate
  report = rigamma (nu / 2, .5);
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Inverse Chi-Squared 
 * deviates according to an input parameter
 *
 * This function generates a Matrix of Inverse 
 * Chi-Squared(\a nu) deviates according to an input parameter.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param nu a constant reference to the \a nu parameter 
 * from the Inverse Chi-Squared distribution.
 * \return the Matrix of random numbers.
 */

Matrix
richisq (const int& rows, const int& cols, const double& nu)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1014:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::richisq(const int& rows, const int& "
         << "cols, const double& nu)" << endl;
    exit (1014);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = richisq (nu);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a random Exponential deviate according 
 * to an input parameter
 *
 * This function generates a random Exponential(\a beta) 
 * deviate according to an input parameter.
 * \param beta a constant reference to the \a beta parameter 
 * from the Exponential distribution.
 * \return the random number.
 */
double
rexp (const double& beta)
{
  static double report;

  // Check for allowable parameter
  if (beta <= 0) {
    cerr << "Error 1015:  inverse scale parameter beta <= 0 in "
         << "SCYTHE::rexp(const double& beta)"
         << endl;
    exit (1015);
  }

  report = -::log (runif ()) / beta;
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Exponential deviates according 
 * to an input parameter
 *
 * This function generates a Matrix of Exponential(\a beta) 
 * deviates according to an input parameter.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param beta a constant reference to the \a beta parameter 
 * from the Exponential distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rexp (const int& rows, const int& cols, const double& beta)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1016:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rexp(const int& rows, const int& "
         << "cols, const double& beta)" << endl;
    exit (1016);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rexp (beta);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a random Student-t deviate according 
 * to input parameters
 *
 * This function generates a random Student-t deviate 
 * according to input parameters.
 * \param mu a constant reference to the \a mu parameter from 
 * the Student-t distribution.
 * \param sigma2 a constant reference to the \a simga^2 parameter 
 * from the Student-t distribution.
 * \param nu a constant reference to the \a nu parameter from 
 * the Student-t distribution.
 * \return the random number.
*/
double
rt (const double& mu, const double& sigma2,
    const double& nu)
{
  static double report;
  double x, z;
  
  // Check for allowable paramters
  if (sigma2 <= 0) {
    cerr << "Error 1017:  variance parameter sigma2 <= 0 in " 
         << "SCYTHE::rt(const double& mu, const double& "
         << " sigma2, const double& nu)" << endl;
    exit (1017);
  }
  if (nu <= 0) {
    cerr << "Error 1018:  d.o.f parameter nu <= 0 in Scythe_Simulate" 
         << ".h::rt(const double& mu, const double& sigma2 const double"
         << "& nu)" << endl;
    exit (1018);
  }

  z = rnorm1 ();
  x = rchisq (nu);
  report = mu + ::sqrt (sigma2) * z * ::sqrt (nu) / ::sqrt (x);
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Student-t deviates 
 * according to input parameters
 *
 * This function generates a Matrix of Student-t deviates 
 * according to input parameters.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param mu a constant reference to the \a mu parameter 
 * from the Student-t distribution.
 * \param sigma2 a constant reference to the \a simga^2 
 * parameter from the Student-t distribution.
 * \param nu a constant reference to the \a nu parameter 
 * from the Student-t distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rt (const int& rows, const int& cols, const double& mu,
    const double& sigma2, const double& nu)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1019:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rt(const int& rows, const int& cols"
         << ", const double& mu, const double& sigma2, const double& "
         << "nu)" << endl;
    exit (1019);
  }
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rt (mu, sigma2, nu);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}


/**********************************************************************/
/*!
 * \brief Generates a random Beta deviate according to input parameters
 *
 * This function generates a random Beta deviate according to 
 * input parameters.  This function uses the Ratio of Chi-Squared rule.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Beta distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Beta distribution.
 * \return the random number.
 */
double
rbeta (const double& alpha, const double& beta)
{
  static double report;
  double xalpha, xbeta;
  
  // Check for allowable parameters
  if (alpha <= 0) {
    cerr << "Error 1020:  alpha parameter <= 0 in SCYTHE::" 
         << "rbeta(const double& alpha, const double& beta)"
         << endl;
    exit (1020);
  }
  if (beta <= 0) {
    cerr << "Error 1021:  beta parameter <= 0 in SCYTHE::" 
         << "rbeta(const double& alpha, const double& beta)"
         << endl;
    exit (1021);
  }
  xalpha = rchisq (2 * alpha);
  xbeta = rchisq (2 * beta);
  report = xalpha / (xalpha + xbeta);
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Beta deviates according 
 * to input parameters
 *
 * This function generates a Matrix of Beta deviates according to 
 * input parameters. This function uses the Ratio of Chi-Squared rule.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Beta distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Beta distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rbeta (const int& rows, const int& cols, const double& alpha,
       const double& beta)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1022:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rbeta(const int& rows, const int& cols"
         << ", const double& alpha, const double& beta)" 
         << endl;
    exit (1022);
	}
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rbeta (alpha, beta);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a random Logistic deviate
 *
 * This function generates a random Logistic deviate, 
 * using a rescaled uniform deviate.
 * \param alpha a constant reference to the \a alpha 
 * parameter from the Logistic distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Logistic distribution.
 * \return the random number.
 */
double
rlogis (const double& alpha, const double& beta)
{
  static double report;
  double unif;
  
  // Check for allowable paramters
  if (beta <= 0) {
    cerr << "Error 1023:  beta parameter <= 0 in SCYTHE::" 
         << "rlogis(const double& alpha, const double& beta)"
         << endl;
    exit (1023);
    }

  unif = runif ();
  report = alpha + beta * ::log (unif / (1 - unif));
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of logistic deviates according 
 * to input parameters
 *
 * This function generates a Matrix of logistic deviates, 
 * using a rescaled uniform deviate.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param alpha a constant reference to the \a alpha parameter 
 * from the Beta distribution.
 * \param beta a constant reference to the \a beta parameter 
 * from the Beta distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rlogis (const int& rows, const int& cols, const double& alpha, 
        const double& beta)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1024:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rlogis(const int& rows, const int& "
         << "cols, const double& alpha, const double& beta)" 
         << endl;
    exit (1024);
	}
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rlogis (alpha, beta);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a random Bernoulli deviate according to an 
 * input parameter
 * This function generates a random Bernoulli deviate according 
 * to an input parameter.
 * \param p a constant reference to the \a p parameter from the 
 * logistic distribution.
 * \return the random number.
 */
int
rbern (const double& p)
{
  static int report;
  double unif;
  
  // Check for allowable paramters
  if (p < 0 || p > 1) {
    cerr << "Error 1025:  p parameter not in [0,1] in SCYTHE" 
         << "::rbern(const double& p)" << endl;
    exit (1025);
  }

  unif = runif ();
  if (unif < p)
    report = 1;
  else
    report = 0;
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of Bernoulli deviates according 
 * to an input parameter
 *
 * This function generates a Matrix of Bernoulli deviates 
 * according to an input parameter.
 * \param rows a constant reference to the number of rows 
 * in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param p a constant reference to the \a p parameter from the 
 * logistic distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rbern (const int& rows, const int& cols, const double& p)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1026:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rbern(const int& rows, const int& " 
         << "cols, const double& p)" 
         << endl;
    exit (1026);
	}
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rbern (p);

  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a random Binomial deviate according to 
 * input parameters
 *
 * This function generates a random Binomial deviate according to 
 * input parameters.  This function draws from a Binomial distribution 
 * with parameter \a p by calling the \a runif() function \a n times, 
 * and counts the number of successes.
 * \param n a constant reference to \a n, the number of tries.
 * \param p a constant reference to the \a p parameter from the 
 * Bernoulli distribution.
 * \return the random Bernoulli deviate.
*/
int
rbin (const int& n, const double& p)
{
  static int report;
  int count, i;
  double hold;
  
  // Check for allowable parameters
  if (n <= 0) {
    cerr << "Error 1027:  n <= 0 in SCYTHE::rbin(" 
         << "const int& n, const double& p)"  << endl;
    exit (1027);
  }
  if (p < 0 || p > 1) {
    cerr << "Error 1028:  p not in [0,1] in SCYTHE::" 
         << "rbin(const int& n, const double& p)"  << endl;
    exit (1028);
  }
  
  // Loop and count successes
  count = 0;
  for (i = 0; i < n; i++) {
    hold = runif ();
    if (hold < p)
      count++;
  }
  report = count;
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of integer random Binomial deviates 
 * according to input parameters
 *
 * This function generates a Matrix of integer random Binomial 
 * deviates according to input parameters.
 * \param rows a constant reference to the number of rows in 
 * the Matrix.
 * \param cols a constant reference to the number of columns in 
 * the Matrix.
 * \param n a constant reference to \a n, the number of tries.
 * \param p a constant reference to the \a p parameter from the 
 * Bernoulli distribution.
 * \return the Matrix of random numbers.
 */
Matrix
rbin (const int& rows, const int& cols, const int& n, const double& p)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1029:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rbin(const int& rows, const int& cols"
         << ", const int& n, const double& p)" 
         << endl;
    exit (1029);
	}
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rbin (n, p);
  
  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Generates a pseudo-random Wishart deviate
 *
 * This function generates a pseudo-random Wishart deviate with \a v 
 * degrees of freedom and scale-matrix \a S.
 * \param v a constant reference to \e v, the number degrees of freedom.
 * \param S a constant reference to the \e S, the scale matrix.
 * \return the Matrix of pseudo-random Wishart deviates.
 */
Matrix
rwish(const int& v, const Matrix& S) {
  if (rows(S) != cols(S)) {
    cerr << "Error 1030: S not square in SCYTHE::"
         << "rwish(const int& v, const Matrix& S)" << endl;
    exit(1030);
  }
  if (v < rows(S)) {
    cerr << "Error 1031: v < dimension of S in SCYTHE::"
         << "rwish(const int& v, const Matrix& S)" << endl;
    exit(1031);
  }
  
  int k = rows(S);
  Matrix A = Matrix(k,k);
  Matrix C = cholesky(S);
  
  for (int i=0; i<v; ++i) {
    Matrix alpha = C*rnorm(k,1);
    A = A + (alpha * t(alpha));
  }
  return(A);
}

/**********************************************************************/
/*!
 * \brief Generates a log-Gamma deviate
 *
 * This function generates a log-Gamma deviate.  This is useful 
 * for computing factorials, and also some statistical functions.
 * \param x a constant reference to \a x, the input parameter.
 * \return the value of the log-Gamma function.
 */
double
gammln (const double& x)
{
  // Check if xx is positive
  if (x <= 0) {
    cerr << "Error 1032: x <= 0 in SCYTHE::"
         << "rgammln(const double& x)" << endl;
    exit (1032);
  }
	double cof[6] = {
		76.18009172947146, -86.50532032941677,
		24.01409824083091, -1.231739572450155,
		0.1208650973866179e-2, -0.5395239384953e-5
	};
	double y = x;
	double tmp = x + 5.5 - (x + 0.5) * ::log(x + 5.5);
	double ser = 1.000000000190015;
	for (int j = 0; j <= 5; j++) {
		ser += (cof[j] / ++y);
	}
	return(::log(2.5066282746310005 * ser / x) - tmp);
}

/**********************************************************************/
/*!
 * \brief Returns the natural log of the Beta function
 *
 * This function returns the natural log of the Beta function.
 * \param x a constant reference to \a x, an input parameter.
 * \param y a constant reference to \a y, an input parameter.
 * \return the value of the log-Beta function.
 */
double
betaln(const double& x, const double& y){
  return gammln(x) + gammln(y) - gammln(x+y);
}

/**********************************************************************/
/*!
 * \brief Returns a Beta function value
 *
 * This function returns a Beta function value.
 * \param x a constant reference to \a x, an input parameter.
 * \param y a constant reference to \a y, an input parameter.
 * \return the value of the Beta function.
 */
double
beta(const double& x, const double& y){
  return ::exp(gammln(x) + gammln(y) - gammln(x+y));
}

/**********************************************************************/
/*!
 * \brief Generates a random variate from the Beta-binomial distribution
 *
 * This function returns a random variate from the Beta-binomial 
 * distribution.
 * \param n, a constant reference to \a n, an input parameter.
 * \param alpha a constant reference to \a alpha, an input parameter.
 * \param beta a constant reference to \a beta, an input parameter.
 * \return the value of the Beta function.
 */
int
rbetabin (const int& n, const double& alpha, const double& beta)
{
  int static report;
  double phi;
  
  // Check for allowable parameters
  if (alpha <= 0) {
    cerr << "Error 1033: alpha <= 0 in SCYTHE::"
         << "rbetabin(const int& n, const double& alpha, const double"
         << "& beta)" << endl;
    exit (1033);
  }
  if (beta <= 0) {
    cerr << "Error 1034: beta <= 0 in SCYTHE::"
         << "rbetabin(const int& n, const double& alpha, const double"
         << "& beta)" << endl;
    exit (1034);
  }
  if (n <= 0) {
    cerr << "Error 1035: n <= 0 in SCYTHE::"
         << "rbetabin(const int& n, const double& alpha, conss double"
         << "& beta)" << endl;
    exit (1035);
  }
  
  phi = rbeta (alpha, beta);
  report = rbin (n, phi);
  return (report);
}

/**********************************************************************/
/*!
 * \brief Generates a Matrix of random variates from the Beta-binomial 
 * distribution
 *
 * This function returns Matrix of random variates from the
 * Beta-binomial distribution.
 * \param rows a constant reference to the number of rows in the Matrix.
 * \param cols a constant reference to the number of columns 
 * in the Matrix.
 * \param n, a constant reference to \a n, an input parameter.
 * \param alpha a constant reference to \a alpha, an input parameter.
 * \param beta a constant reference to \a beta, an input parameter.
 * \return the Matrix of values of the Beta-binomial distribution.
 */
Matrix
rbetabin (const int& rows, const int& cols, const int& n, 
          const double& alpha, const double& beta)
{
  int size = rows * cols;
  if (size <= 0) {
    cerr << "Error 1036:  Attempted to create Matrix of size <= 0 in "
         << "SCYTHE::rbetabin(const int& rows, const int&" 
         << " cols, const int& n, const double& alpha, const double&"
         << " beta)" 
         << endl;
    exit (1036);
	}
  double *newdata = new double[size];
  int i;
  for (i = 0; i < size; ++i)
    newdata[i] = rbetabin (n, alpha, beta);
  
  Matrix temp = Matrix (newdata, rows, cols);
  delete[]newdata;
  return temp;
}

/**********************************************************************/
/*!
 * \brief Returns the univariate standard normal density evaluated 
 * at \a x
 *
 * Returns the univariate standard normal density evaluated at \a x.
 * \param x a constant reference to the \a x parameter.
 * \return the value of the standard normal density.
*/
double
dnorm1 (const double& x)
{
  double f;
  f = (1.0 / (::sqrt (2.0 * PI))) * ::exp (-0.5 * (x * x));
  return (f);
}

/**********************************************************************/
/*!
 * \brief Returns the natural log of the multivariate normal density
 *
 * Returns the natural log of the multivariate normal density 
 * with mean \a mu and covariance matrix \a sigma evaluated at \a x.  
 * (\a mu and \a x assumed to be column vectors).
 * \param x a constant reference to the \a x parameter, a column vector.
 * \param mu a constant reference to the \a mu parameter, the mean 
 * (a column vector).
 * \param sigma a constant reference to the \a sigma parameter, 
 * the covariance matrix.
 * \return the value of the natural log of the multivariate 
 * normal density.
 */
double
lndmvn (const Matrix& x, const Matrix& mu, 
               const Matrix& Sigma){
  if (cols(x) != 1) {
    cerr << "Error 1037: x not column vector in SCYTHE::"
         << "lndmvn(const Matrix& x, const Matrix& mu, const Matrix&"
         << " Sigma)" << endl;
    exit(1037);
  }
  if (cols(mu) != 1) {
    cerr << "Error 1038: mu not column vector in SCYTHE::"
         << "lndmvn(const Matrix& x, const Matrix& mu, const Matrix&"
         << " Sigma)" << endl;
    exit(1038);
  }
  if (rows(Sigma) != cols(Sigma)) {
    cerr << "Error 1039: Sigma not square in SCYTHE::"
         << "lndmvn(const Matrix& x, const Matrix& mu, const Matrix&"
         << " Sigma)" << endl;
    exit(1039);
  }
  if (rows(mu) != rows(Sigma)){
    cerr << "Error 1040: mu & Sigma not same dimension in Scythe_"
         << "Simulate.h::lndmvn(const Matrix& x, const Matrix& mu," 
         << " const Matrix& Sigma)" << endl;
    exit(1040);
  }
  if (rows(mu) != rows(x)){
    cerr << "Error 1041: mu and x not of same dimension in Scythe_"
         << "Simulate.h::lndmvn(const Matrix& x, const Matrix& mu," 
         << " const Matrix& Sigma)" << endl;
    exit(1041);
  }
  int k = rows(mu);
  return ( (-k/2.0)*::log(2*PI) -0.5 * ::log(det(Sigma)) 
          -0.5*t(x-mu)*invpd(Sigma)*(x-mu) )[0];
}

/**********************************************************************/
/*!
 * \brief Returns the univariate standard normal cumulative 
 * distribution function (CDF)
 *
 * Returns the univariate standard normal cumulative distribution 
 * function (CDF) evaluated at \a x.
 * \param xx a constant reference to the \a x parameter.
 * \param eps a constant reference to the \a eps parameter.
 * \param maxit a constant reference to the \a maxit parameter, which
 * is the maximum number of iterations this loop will run until it 
 * converges.
 * \param minit a constant reference to the \a minit parameter, which 
 * reflects the minimum number of iterations this loop will run.
 * \return the value of the univariate standard normal CDF.
 */
double
pnorm1 (const double& xx, const double& eps, const double& maxit,
	const double& minit){
  double x = xx;
  int i;
  
  if (x == 0)
    return 0.5;
  
  // if x nonnegative keep sign of x and return f + 0.5
  if (x > 0) {
    double a = (1 / (::sqrt (2.0 * PI))) * ::exp (-(x * x) / 2.0) * x;
    double f = a;
    for (i = 1; i < maxit; ++i) {
      a = a * (x * x) / (2 * i + 1);
      f += a;
      if (::fabs(a)/f < eps && i > minit)
        break;
    }
    if (i == maxit) {
      cerr << "Error 1042: x (" << x << ") did not converge in "
           << "SCYTHE::pnorm1(double x)" << endl;
      exit (1042);
    } else
      return (f + 0.5);
  } else {
    // if x negative flip the sign and return 0.5 - f
    x = -1*x;
    double a = (1 / (::sqrt (2.0 * PI))) * ::exp (-(x * x) / 2.0) * x;
    double f = a;
    for (i = 1; i < maxit; ++i) {
      a = a * (x * x) / (2 * i + 1);
      f += a;
      if (::fabs(a)/f < eps && i > minit)
        break;
    }
    // Makes sense to have 2 error codes here for debug
    if (i == maxit) {
      cerr << "Error 1043: x (" << x << ") did not converge in "
           << "SCYTHE::pnorm1(double x)" << endl;
      exit (1043);
    } else
      return (0.5 - f);
  }
}

/**********************************************************************/
/*!
 * \brief Returns the quantile of the standard normal distribution 
 * associated with a given probability \a p
 *
 * Returns the quantile of the standard normal distribution associated 
 * with a given probability \a p.
 * \param in_p a constant reference to the \a p parameter, which is 
 * the given probability.
 * \return the quartile of the standard normal distribution 
 * associated with the given probability.
 */
double
qnorm1 (const double& in_p)
{
  double lim = 10e-20;
  double p0 = -0.322232431088;
  double q0 = 0.0993484626060;
  double p1 = -1.0;
  double q1 = 0.588581570495;
  double p2 = -0.342242088547;
  double q2 = 0.531103462366;
  double p3 = -0.0204231210245;
  double q3 = 0.103537752850;
  double p4 = -0.453642210148e-4;
  double q4 = 0.38560700634e-2;
  double xp = 0.0;
  double p = in_p;
  
  if (p > 0.5)
    p = 1 - p;
    
  if (p < lim) {
    cerr << "Error 1044: p outside accuracy limit in Scythe_"
         << "Simulate.h::qnorm1(const double& in_p)" << endl;
    exit (1044);
  }
  
  if (p == 0.5)
    return xp;
  
  double y = ::sqrt (::log (1.0 / ::pow (p, 2)));
  xp = y + ((((y * p4 + p3) * y + p2) * y + p1) * y + p0) /
       ((((y * q4 + q3) * y + q2) * y + q1) * y + q0);
  
  if (in_p < 0.5)
    xp = -1 * xp;
  return xp;
}

/**********************************************************************/
/*!
 *\brief Returns the univariate Student-t density evaluated at \a x 
 * with mean \a mu, scale \a sigma^2, and \a nu degrees of freedom
 *
 * Returns the univariate Student-t density evaluated at \a x with 
 * mean \a mu, scale \a sigma^2, and \a nu degrees of freedom.
 * \param x a constant reference to the \a x parameter, which is 
 * the point at which the Student-t distribution will be evaluated.
 * \param mu a constant reference to the \a mu parameter, which 
 * reflects the mean of the Student-t distribution.
 * \param sigma2 a constant reference to the \a sigma^2 parameter, 
 * which reflects the scale or variance of the Student-t distribution.
 * \param nu a constant reference to the \a nu parameter, which 
 * reflects the degrees of freedom for the Student-t distribution
 * \return the univariate Student-t density evaluated at \a x.
 */
double
dt1(const double& x, const double& mu, const double& sigma2, 
    const double& nu){  
  double logdens = gammln((nu+1.0)/2.0) - ::log(::sqrt(nu*PI)) - 
                   gammln(nu/2.0) - ::log(::sqrt(sigma2)) - (nu+1.0)/2.0 * 
                   ::log(1.0 + (::pow((x-mu),2.0))/(nu * sigma2));
  return(::exp(logdens));
}

/**********************************************************************/
/*!
 *\brief Returns the natural log of the univariate Student-t density 
 * evaluated at \a x with mean \a mu, scale \a sigma^2, and \a nu 
 * degrees of freedom
 *
 * Returns the univariate student-t density evaluated at \a x with 
 * mean \a mu, scale \a sigma^2, and \a nu degrees of freedom.
 * \param x a constant reference to the \a x parameter, which is 
 * the point at which the Student-t distribution will be evaluated.
 * \param mu a constant reference to the \a mu parameter, which 
 * reflects the mean of the Student-t distribution.
 * \param sigma2 a constant reference to the \a sigma^2 parameter, 
 * which reflects the scale or variance of the Student-t distribution.
 * \param nu a constant reference to the \a nu parameter, which 
 * reflects the degrees of freedom for the Student-t distribution.
 * \return the natural log of the univariate Student-t density 
 * evaluated at \a x.
*/
double 
logdt1(const double& x, const double& mu, const double& sigma2, 
       const double& nu){
  double logdens = gammln((nu+1.0)/2.0) - ::log(::sqrt(nu*PI)) - 
                   gammln(nu/2.0) - ::log(::sqrt(sigma2)) - (nu+1.0)/2.0 * 
                   ::log(1.0 + (::pow((x-mu),2.0))/(nu * sigma2));
  return(logdens);
}

/**********************************************************************/
/*!
 * \brief Returns the natural log of the ordinate of the Beta density
 * evaluated at \a x with Shape1 \a a, and Shape2 \a b
 *
 * Returns the natural log of the ordinate of the Beta density evaluated
 * at \a x with shape1 \a a, and shape2 \a b.
 * \param x a constant reference to the \a x parameter, 
 * which is the point at which the Beta desnsity will be evaluated.
 * \param a a constant reference to the \a shape 1 parameter.
 * \param b a constant reference to the \a shape 2 parameter.
 * \return the natural log of the ordinate of the Beta density evaluated
 * at \a x.
 */
double
logdbeta1(const double& x, const double& a, const double& b) {  
  if ((x<0.0) || (x>1.0)) {
    cerr << "Error 1045: x not in [0,1] in Scythe_Simulate.j::"
         << "logdbeta1(const double& x, const double& a, const" 
         << " double& b)" << endl;
    exit(1045);
  }
	if (a < 0.0) {
    cerr << "Error 1046: a < 0 in Scythe_Simulate.j::"
         << "logdbeta1(const double& x, const double& a, const" 
         << " double& b)" << endl;
    exit(1046);
	}
	if (b < 0.0) {
    cerr << "Error 1047: b < 0 in Scythe_Simulate.j::"
         << "logdbeta1(const double& x, const double& a, const" 
         << " double& b)" << endl;
    exit(1047);
	}
		
  return (a-1.0) * ::log(x) + (b-1) * ::log(1.0-x) - betaln(a,b);
}

/**********************************************************************/
/*!
 * \brief Returns the univariate Beta density evaluated at \a x 
 * with Shape1 \a a, and Shape 2 \a b
 *
 * Returns the univariate Beta density evaluated at \a x with shape1 
 * \a a, and shape2 \a b.
 * \param x a constant reference to the \a x parameter, which is 
 * the point at which the Beta desnsity will be evaluated.
 * \param a a constant reference to the \a shape 1 parameter.
 * \param b a constant reference to the \a shape 2 parameter.
 * \return the natural log of the univariate Beta density evaluated 
 * at \a x.
 */
double
dbeta1(const double& x, const double& a, const double& b){  
  if ((x<0.0) || (x>1.0)) {
    cerr << "Error 1048: x not in [0,1] in SCYTHE::"
         << "dbeta1(const double& x, const double& a, const" 
         << " double& b)" << endl;
    exit(1048);
  }
	if (a < 0.0) {
    cerr << "Error 1049: a < 0 in SCYTHE::"
         << "dbeta1(const double& x, const double& a, const" 
         << " double& b)" << endl;
    exit(1049);
	}
	if (b < 0.0) {
    cerr << "Error 1050: b < 0 in SCYTHE::"
         << "dbeta1(const double& x, const double& a, const" 
         << " double& b)" << endl;
    exit(1050);
	}
  
  return (::pow(x, (a-1.0)) * ::pow((1.0-x), (b-1.0)) ) / beta(a,b);
}

/**********************************************************************/
/*!
 * \brief Simulates from a truncated Normal distribution
 *
 * This function simulates from a truncated Normal distribution 
 * with mean parameter \a m, variance parameter \a v, on the 
 * interval \a (below,above).
 * This function uses the Inverse CDF method.
 * \param m a constant reference to the \a m parameter, 
 * which is the mean.
 * \param v a constant reference to the \a v parameter, 
 * which is the variance.
 * \param below a constant reference to the \e below 
 * parameter, the lower bound of the interval.
 * \param above a constant reference to the \e above 
 * parameter, the upper bound of the interval.
 * \return the random value from the truncated Normal distribution.
 */
double 
rtnorm(const double& m, const double& v, const double& below, 
       const double& above){  
  if ( (below > above) > 0){
    cerr << "Error 1051: truncation bounds not logically consistent in " 
         << "SCYTHE::rtnorm(const double& m const double& "
         << "v, const double& below, const double& above)" << endl;
    exit(1051); 
  }

  double FA = 0.0;
  double FB = 0.0;
  double sd = ::sqrt(v);
  if ( (::fabs((above-m)/sd) < 6.36)  && (::fabs((below-m)/sd) < 6.36)){
    FA = pnorm1((above-m)/sd);
    FB = pnorm1((below-m)/sd);    
  }
  if ((((above-m)/sd) < 6.36)  && (((below-m)/sd) <= -6.36) ){ 
    FA = pnorm1((above-m)/sd);
    FB = 0.0;
  }
  if ( (((above-m)/sd) >= 6.36)  && (((below-m)/sd) > -6.36) ){ 
    FA = 1;
    FB = FB = pnorm1((below-m)/sd);    
  } 
  if ( (::fabs((above-m)/sd) > 6.36)  && (::fabs((below-m)/sd) > 6.36)){
    cerr << "ERROR 1054: truncation bounds too far from 0 in " <<
      "SCYTHE::rtnorm(const double& m const double& "<<
      "v, const double& below, const double& above) " << 
      "try using SCYTHE::rtanorm_combo() or "<<
      "SCYTHE::rtbnorm_combo() " <<endl;
    exit(1054);      
  }
  
  double term = runif()*(FA-FB)+FB;
  
  if (term < 1e-10)
    term = 1e-10;
  
  if (term > (1 - 1e-10))
    term = 1 - 1e-10;
  
  double draw = m + sd * qnorm1(term);
  
  if (draw > above)
    draw = above;
  
  if (draw < below)
    draw = below;
  
  return draw;
}



/**********************************************************************/
/*!
 * \brief Sample from a truncated Normal distribution
 *
 * Sample from a Normal distribution with mean \a m, variance \a v, 
 * truncated to the interval \a (below,infinity) via the slice sampler
 * of Robert and Casella (1999), pp. 288-289. Sampler is run for \a 
 * iter iterations.  If \a iter is not specified, the default amount 
 * of iterations is 10.  NOTE:  THIS METHOD ONLY WORKS WHEN \a 
 * below >= \e m.
 * \param m a constant reference to the \a m parameter, 
 * which is the mean.
 * \param v a constant reference to the \a v parameter, 
 * which is the variance.
 * \param below a constant reference to the \a below parameter, the lower
 * bound of the interval.
 * \param iter a constant reference to the \a iter parameter, the number
 * of iterations the sampler will run.
 * \return the random value from the truncated Normal distribution.
 */
double rtbnorm_slice(const double& m, const double& v, const double& below, 
	      const int& iter=10){
  
  if (below < m){
    cerr << "ERROR 1055: truncation point < mean in "<<
      "SCYTHE::rtbnorm_slice(const double&, const double&, " << 
      "const double&, const int& )" << endl;
      exit(1055); 
  }

  if (v <= 0){
    cerr << "ERROR 1056: variance non-positive in " <<
      "SCYTHE::rtbnorm_slice(const double&, const double&, " << 
      "const double&, const int& )" << endl;
      exit(1056); 
  }
  
  double z = 0;
  double x = below + .00001;
  
  for (int i=0; i<iter; ++i){
    z = runif()*::exp(-1*::pow((x-m),2)/(2*v));
    x = runif()*( (m + ::sqrt(-2*v*::log(z))) - below) + below;
  }
  if (isinf(x)){
    cerr << "WARNING: mean extremely far from truncation point in " << 
      "SCYTHE::rtbnorm_slice(const double&, const double&, " << 
      "const double&, const int& ).  Returning truncation point. " << endl;
    return below; 
  }
  return x;
}




/**********************************************************************/
/*!
 * \brief Sample from a truncated Normal distribution
 *
 * Sample from a Normal distribution with mean \a m, variance \a v, 
 * truncated to the interval \a (-infinity,above) via the slice sampler
 * of Robert and Casella (1999), pp. 288-289. Sampler is run for \a iter 
 * iterations.  If \a iter is not specified, the default amount of 
 * iterations is 10.
 * \param m a constant reference to the \a m parameter, which is 
 * the mean.
 * \param v a constant reference to the \a v parameter, which is the
 * variance.
 * \param above a constant reference to the \a above parameter, the upper
 * bound of the interval.
 * \param iter a constant reference to the \a iter parameter, the number
 * of iterations the sampler will run.
 * \return the random value from the truncated Normal distribution.
 */
double rtanorm_slice(const double& m, const double& v, const double& above, 
	      const int& iter=10){
  
  if (above > m){
      cerr << "ERROR 1057: truncation point > mean in " << 
      "SCYTHE::rtanorm_slice(const double&, const double&, " << 
      "const double&, const int& )" << endl;
      exit(1057); 
  }

  if (v <= 0){
      cerr << "ERROR 1058: variance non-positive in " << 
      "SCYTHE::rtanorm_slice(const double&, const double&, " << 
      "const double&, const int& )" << endl;
      exit(1058); 
  }
  double below = -1*above;
  double newmu = -1*m;
  double z = 0;
  double x = below + .00001;
  
  for (int i=0; i<iter; ++i){
    z = runif()*::exp(-1*::pow((x-newmu),2)/(2*v));
    x = runif()*( (newmu + ::sqrt(-2*v*::log(z))) - below) + below;
  }
  if (isinf(x)){
    cerr << "WARNING: mean extremely far from truncation point in " << 
      "SCYTHE::rtanorm_slice(const double&, const double&, " << 
      "const double&, const int& ).  Returning truncation point. " << endl;
    return above; 
  }
  return -1*x;
  
}

/**********************************************************************/
/*!
 * \brief Sample from a truncated Normal distribution
 *
 * Sample from a Normal distribution with mean \a m, variance \a v, 
 * truncated to the interval \a (below,infinity) using a combination of 
 * rejection sampling (when \em \e>= \e below) and the slice sampling 
 * method of Robert and Casella (1999), pp. 288-289. 
 * Sampler is run for \a iter iterations.  If \a iter is not specified, 
 * the default amount of iterations is 10.
 * \param m a constant reference to the \a m parameter, 
 * which is the mean.
 * \param v a constant reference to the \a v parameter, which is the
 * variance.
 * \param below a constant reference to the \a below parameter, the lower
 * bound of the interval.
 * \param iter a constant reference to the \a iter parameter, the number
 * of iterations the sampler will run.
 * \return the random value from the truncated Normal distribution.
 */
double rtbnorm_combo(const double& m, const double& v, const double& below,
		     const int& iter=10){

  if (v <= 0){
    cerr << "ERROR 1059: variance non-positive in " << 
      "SCYTHE::rtbnorm_comba(const double&, const double&, " << 
      "const double&, const int& )" << endl;
    exit(1059); 
  }

  // do rejection sampling and return value
  if (m >= below){
    double x = rnorm(m, v);
    while (x < below)
      x = rnorm(m,v);
    return x;
  }
  // do slice sampling and return value
  else{
    double z = 0;
    double x = below + .00001;
    for (int i=0; i<iter; ++i){
      z = runif()*::exp(-1*::pow((x-m),2)/(2*v));
      x = runif()*( (m + ::sqrt(-2*v*::log(z))) - below) + below;
    }
    if (isinf(x)){
      cerr << "WARNING: mean extremely far from truncation point in " << 
	"SCYTHE::rtbnorm_comba(const double&, const double&, " << 
	"const double&, const int& ).  Returning truncation point. " << endl;
      return below; 
    }
    return x;
  }
}


/**********************************************************************/
/*!
 * \brief Sample from a truncated Normal distribution
 * Sample from a Normal distribution with mean \a m, variance \a v, 
 * truncated to the interval \a (-infinity,above) using a combination
 * of rejection sampling (when \a m <= \a above) and  the slice sampler
 * of Robert and Casella (1999), pp. 288-289. Sampler is run for \a iter 
 * iterations.  If \a iter is not specified, the default amount 
 * of iterations is 10.
 * \param m a constant reference to the \a m parameter, 
 * which is the mean.
 * \param v a constant reference to the \a v parameter, 
 * which is the variance.
 * \param above a constant reference to the \a above parameter, the upper
 * bound of the interval.
 * \param iter a constant reference to the \a iter parameter, the number
 * of iterations the sampler will run.
 * \return the random value from the truncated Normal distribution.
 */
double rtanorm_combo(const double& m, const double& v, const double& above,
		     const int& iter=10){

  if (v <= 0){
    cerr << "ERROR 1060: variance non-positive in " <<
      "SCYTHE::rtanorm_comba(const double&, const double&, " << 
      "const double&, const int& )" << endl;
    exit(1060); 
  }

  // do rejection sampling and return value
  if (m <= above){
    double x = rnorm(m, v);
    while (x > above)
      x = rnorm(m,v);
    return x;
  }
  // do slice sampling and return value
  else{
    double below = -1*above;
    double newmu = -1*m;
    double z = 0;
    double x = below + .00001;
    
    for (int i=0; i<iter; ++i){
      z = runif()*::exp(-1*::pow((x-newmu),2)/(2*v));
      x = runif()*( (newmu + ::sqrt(-2*v*::log(z))) - below) + below;
    }
    if (isinf(x)){
      cerr << "WARNING: mean extremely far from truncation point in " << 
	"SCYTHE::rtanorm_combo(const double&, const double&, " << 
	"const double&, const int& ).  Returning truncation point. " << endl;
      return above; 
    }
    return -1*x;
  }
}


/**********************************************************************/
/*!
 * \brief Computes the factorial \a n! of a non-negative integer
 * Computes the factorial \a n! of a non-negative integer.
 * \param n a constant reference to \a n, the s the mean.
 * \return the factorial of the number \a n.
 */
int factorial(int n) {
  if (n < 0){
    cerr << "Error 1052: negative integer passed to SCYTHE::"
         << "factorial(int n)" << endl;
    exit(1052); 
  }
  if (n == 0)
    return 1;
  else 
    return n * factorial(n-1);
}

/**********************************************************************/
/*!
 * \brief Computes the natural log of the factorial of a non-negative
 *  integer
 * Computes the natural log of the factorial \a n! of a non-negative 
 * integer.
 * \param n .
 * \return the log factorial of the number \a n.
 */
double logfact(const int& n) {
  // Check if n is non-negative
  if (n < 0) {
    cerr << "Error ????: n < 0 in SCYTHE::"
         << "logfact(const int& x)" << endl;
    exit (1032);
  }
  
  double x = n+1;
  double cof[6] = {
    76.18009172947146, -86.50532032941677,
    24.01409824083091, -1.231739572450155,
    0.1208650973866179e-2, -0.5395239384953e-5
  };
  double y = x;
  double tmp = x + 5.5 - (x + 0.5) * ::log(x + 5.5);
  double ser = 1.000000000190015;
  for (int j = 0; j <= 5; j++) {
    ser += (cof[j] / ++y);
  }
  return(::log(2.5066282746310005 * ser / x) - tmp);
}


/**********************************************************************/
/*!
 * \brief Computes the Poisson probability mass function
 * \param x the quantile at which the Poission probabilty function
 * is to be evaluated
 * \param lambda the mean and variance of the Poisson distribution 
 * \return the value of the Poisson mass function at \a x
 */
double dpois(const int& x, const double& lambda){
  
  // Check if xx is non-negative
  if (x < 0) {
    cerr << "Error ????: n < 0 in SCYTHE::"
         << "dpois(const int& x, const double& lambda)" << endl;
    exit (50000);
  }
  
  // Check if lambda is positive
  if (lambda <= 0) {
    cerr << "Error ????: lambda <= 0 in SCYTHE::"
         << "dpois(const int& x, const double& lambda)" << endl;
    exit (500000);
  }

  
  // compute log(x!)
  double xx = x+1;
  double cof[6] = {
    76.18009172947146, -86.50532032941677,
    24.01409824083091, -1.231739572450155,
    0.1208650973866179e-2, -0.5395239384953e-5
  };
  double y = xx;
  double tmp = xx + 5.5 - (xx + 0.5) * ::log(xx + 5.5);
  double ser = 1.000000000190015;
  for (int j = 0; j <= 5; j++) {
    ser += (cof[j] / ++y);
  }
  double logfactx = ::log(2.5066282746310005 * ser / xx) - tmp;
  
  return (::exp( -1*logfactx + x * ::log(lambda) - lambda));

}



/**********************************************************************/
/*!
 * \brief Computes the natural log of the Poisson probability mass function
 * \param x the quantile at which the Poission probabilty function
 * is to be evaluated
 * \param lambda the mean and variance of the Poisson distribution 
 * \return the value of the natural log of the Poisson mass 
 * function at \a x
 */
double logdpois(const int& x, const double& lambda){
  
  // Check if xx is non-negative
  if (x < 0) {
    cerr << "Error ????: n < 0 in SCYTHE::"
         << "dpois(const int& x, const double& lambda)" << endl;
    exit (50000);
  }
  
  // Check if lambda is positive
  if (lambda <= 0) {
    cerr << "Error ????: lambda <= 0 in SCYTHE::"
         << "dpois(const int& x, const double& lambda)" << endl;
    exit (500000);
  }

  
  // compute log(x!)
  double xx = x+1;
  double cof[6] = {
    76.18009172947146, -86.50532032941677,
    24.01409824083091, -1.231739572450155,
    0.1208650973866179e-2, -0.5395239384953e-5
  };
  double y = xx;
  double tmp = xx + 5.5 - (xx + 0.5) * ::log(xx + 5.5);
  double ser = 1.000000000190015;
  for (int j = 0; j <= 5; j++) {
    ser += (cof[j] / ++y);
  }
  double logfactx = ::log(2.5066282746310005 * ser / xx) - tmp;
  
  return ( -1*logfactx + x * ::log(lambda) - lambda);

}


/**********************************************************************/
/*!
 * \brief Returns a scalar Poisson pseudo-random deviate using Atkinson's
 * method (see Robert and Casella p. 56
 * \param lambda a constant reference to a double, the Poisson rate 
 * parameter.
 * \return A pseudo-random Poisson deviate
 */
int rpois(const double& lambda){
  // Check if lambda is positive
  if (lambda <= 0) {
    cerr << "Error ????: lambda <= 0 in SCYTHE::"
         << "rpois(const double& lambda)" << endl;
    exit (500000);
  }

  int n;
  
  if (lambda < 33){
    double cutoff = ::exp(-lambda);
    n = -1;
    double t = 1.0;
    do {
      ++n;
      t *= runif();
    } while (t > cutoff);    
  }
  else{
    int accept = 0;
    
    double c = 0.767 - 3.36/lambda;
    double beta = PI/::sqrt(3*lambda);
    double alpha = lambda*beta;
    double k = ::log(c) - lambda - ::log(beta);
    
    while (accept == 0){
      double u1 = runif();
      double x = (alpha - ::log((1-u1)/u1))/beta;
      while (x <= -0.5){
	u1 = runif();
	x = (alpha - ::log((1-u1)/u1))/beta;
      } 
      n = static_cast<int>(x+0.5);
      double u2 = runif();
      
      double lhs = alpha - beta*x + ::log(u2/::pow(1+::exp(alpha-beta*x),2));
      double rhs = k + n*::log(lambda) - logfact(n);
      if (lhs <= rhs)
	accept = 1;
    }
  }
  return n;
  
}



/**********************************************************************/
/*!
 * \brief Sorts a column Matrix (vector) (very slowly)
 * Sorts a column vector (very slowly) and returns the sorted 
 * Matrix (vector).
 * \param A a constant reference to \a A, the vector to be sorted.
 * \return the sorted Matrix (vector).
 */
Matrix
bubbleSort(const Matrix& A) {
  if(cols(A) > 1) {
    cerr << "Error 1053: Matrix (not column vector) passed to"
         << "SCYTHE::bubblesort(const Matrix& A)" << endl;
    exit(1053);
  }
  int ra = rows(A);
  Matrix sorted = A;
  for(int k=ra-1; k>0; k--) {
    for(int i=0; i<k; ++i) {
      if(sorted(i,0)>sorted(i+1,0)) {
        double tmp=sorted(i,0);
        sorted(i,0)=sorted(i+1,0);
        sorted(i+1,0)=tmp;
      }
    }
  }
  return sorted;
}

} // namespace dec
#endif /* SCYTHE_SIMULATE_CC */
