/* Scythe_Simulate.h */
// Function declarations for probability density/distribution functions 
// and pseudo-randon number generators for the Scythe Library
//
// Scythe C++ Library
// Copyright (C) 2000 Kevin M. Quinn and Andrew D. Martin
//
// This code written by:
//
// Kevin Quinn
// Assistant Professor
// Dept. of Political Science and 
// Center for Statistics and the Social Sciences
// Box 354322
// University of Washington
// Seattle, WA  98195-4322
// quinn@stat.washington.edu
//
// Andrew D. Martin
// Assistant Professor
// Dept. of Political Science
// Campus Box 1063
// Washington University
// St. Louis, MO 63130
// admartin@artsci.wustl.edu
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation; either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA

#ifndef SCYTHE_SIMULATE_H
#define SCYTHE_SIMULATE_H

#include <math.h>
#include <limits.h>
#include "Scythe_Double_Matrix.h"

/* Defined Numbers */
#define PI  3.141592653589793238462643383279502884197169399375

// Avoid NameSpace Pollution
namespace SCYTHE {

/**********************************************************************/
//  FUNCTION:  min():  finds the minimum of two doubles
//  Input:  double&, double&
//  Return: double
//  Usage:  minXY = min(x, y);
//  Errors: none
//  Dependencies:

double min (const double& a, const double& b);

/**********************************************************************/
//  FUNCTION:  max():  find the maximum of two doubles
//  Input:  double&, double&
//  Return: double
//  Usage:  maxXY = max(x, y);
//  Errors: none
//  Dependencies:

double max (const double& a, const double& b);

/**********************************************************************/
//  FUNCTION:  setRandSeed():  set the random seed for the pseudo-
//    random number generator(s)
//  Input:  double
//  Return: void
//  Usage:  setRandSeed(s);
//  Errors: none
//  Dependencies:  none

void setRandSeed(const unsigned long&, const unsigned long &,
                 const unsigned long&, const unsigned long &);

/**********************************************************************/
//  FUNCTION:  ranmars():  generates a random uniform (0,1) deviate
//    by a combonation of Marsaglia's KISS and SWB algorithms.
//  Input:  void
//  Return: double
//  Usage:  x = ranmars();
//  Errors: none
//  Dependencies:  limits.h;
//  References:  Based on postings by George Marsaglia and others
//    in 1999 to the USENET groups sci.stat.math, sci.math,
//    sci.math.num-analysis, sci.crypt, sci.physics.

double ranmars();

/**********************************************************************/
//  FUNCTION:  runif():  generates a random uniform number.  In this 
//    application it only calls the Marsaglia generator, but other 
//    versions can be substituted.  All subsequent uniform calls point 
//    here, so changes here will affect all other generators.
//  Input:  void
//  Return: double (pseudo-random uniform deviate)
//  Usage:  x = runif()
//  Errors: none
//  Dependencies: math.h 
//    
//  7/10/2000 (ADM)

inline double runif (void)
{
  return (ranmars());
};

/**********************************************************************/
//  FUNCTION:  runif():  generates a Double_Matrix initialized with 
//  random uniform numbers.  
//  Input:  int&, int&
//  Return: Matrix (of pseudo-random uniform deviates)
//  Usage:  X = runif(row, cols)
//  Errors: 1001   
//  Dependencies: math.h, Scythe_Double_Matrix.h
//    
//  7/10/2000 (ADM)

Matrix runif (const int& rows, const int& cols);

/**********************************************************************/
//  FUNCTION:  rnorm1(): returns a normal (0,1) deviate generated 
//    according to the Box-Muller method. This function uses the 
//    uniform deviates generated in runif()
//  Input:  void
//  Return: double (pseudo-random normal (0,1) deviate) 
//  Usage:  x = rnorm1();
//  Errors: none
//  Dependencies: math.h, runif()
//  References: This code is based on Algorithm 3.1 of Gentle, James E.
//  1998. Random Number Generation and Monte Carlo Methods. Springer
//
//  6/10/2000 (KQ)

double rnorm1 (void);

/**********************************************************************/
//  FUNCTION:  rnorm(): returns a normal(mu,sigma2) deviate.  The 
//    function uses the standard normal generator above rnorm1().  mu 
//    is the mean, and sigma2 is the variance. Note that both have to 
//    be passed as doubles.  For both the scalar and matrix generator,
//    the default mean and variance are zero and one respectively.
//  Input:  double&, double&
//  Return: double (pseudo-random normal(mu,sigma2) deviate)
//  Usage:  x = rnorm(mu,sigma);
//  Errors: 1002
//  Dependencies: math.h, rnorm1(), runif()
//
//  7/10/2000 (ADM)

double rnorm (const double& mu = 0.0, const double& sigma2 = 1.0);

/**********************************************************************/
//  FUNCTION:  rnorm(): returns a matrix initialized with 
//    normal(mu,sigma2) deviate.  See double rnorm() above for details.
//  Input:  int&, int&, double&, double&
//  Return: Matrix
//  Usage:  X = rnorm(rows, cols, mu, sigma);
//  Errors: 1003
//  Dependencies:  math.h, Scythe_Double_Matrix.h, rnorm1(), runif()
//
// 7/10/2000 (ADM)

Matrix rnorm (const int& rows, const int& cols, const double& mu = 0.0,
       const double& sigma2 = 1.0);

/**********************************************************************/
//  FUNCTION:  rgamma1():  returns a gamma(alpha,1) deviate using
//    Best's (1978) simulator as documented in Robert and Cassella 
//    (1999).  The R&S notation is used in the function.  This is used 
//    by rgamma() to generate gamma deviates for all possible
//    parameters.  Note that the gamma(alpha,1) distribution is also 
//    known as the Erlang distribution.
//  Input:  double&
//  Return: double
//  Usage:  x = rgamma1(alpha);
//  Errors: 1004
//  Dependencies:  math.h, runif()
//
// 6/12/2000 (ADM)

double rgamma1 (const double& alpha);

/**********************************************************************/
//  FUNCTION:  rgamma():  returns a gamma(alpha,beta) deviate.  The
//    function uses Best's (1978) simulator as documented in Robert and
//    Casella (1999).  The R&S notation is used in the function.  The
//    function first generates the gamma(alpha,1) deviate, and then 
//    reports the gamma(alpha,beta) deviate using the multiplication
//    rule.
//  Input:  double&, double&
//  Return: double
//  Usage:  x = rgamma(alpha,beta);
//  Errors: 1005-1006
//  Dependencies:  math.h, runif(), rgamma1()
//  
//  7/10/2000 (ADM)

double rgamma (const double& alpha, const double& beta);

/**********************************************************************/
//  FUNCTION:  rgamma():  returns a matrix of gamma(alpha,beta)
//    deviates.  See double gamma(alpha,beta) above for details.
//  Input:  int&, int&, double&, double&
//  Return: Matrix
//  Usage:  X = rgamma(rows, cols, alpha,beta);
//  Errors: 1007
//  Dependencies:  math.h, Scythe_Double_Matrix.h, runif(), rgamma1()
//  
//  7/10/2000 (ADM)

Matrix rgamma (const int& rows, const int& cols, const double& alpha, 
        const double& beta);

/**********************************************************************/
//  FUNCTION:  rigamma():  returns a random inverse-gamma deviate.  It
//    merely calls the rgamma() function, and takes the reciprocal.
//  Input:  double&, double&
//  Return: double
//  Usage:  x = rigamma(alpha,beta);
//  Errors: 1008-1009
//  Dependencies:  math.h, rgamma(), rgamma1(), runif()
// 
// 7/10/2000 (ADM)

double rigamma (const double& alpha, const double& beta);

/**********************************************************************/
//  FUNCTION:  rigamma():  returns a random a matrix of inverse-gamma
//    deviates.  
//  Input:  int&, int&, double&, double&
//  Return: Matrix
//  Usage:  X = rigamma(rows, cols, alpha,beta);
//  Errors: 1010
//  Dependencies:  math.h, Scythe_Double_Matrix.h 
//                rgamma(), rgamma1(), runif()
// 
// 7/10/2000 (ADM)

Matrix rigamma (const int& rows, const int& cols, const double& alpha,
   const double& beta);

/**********************************************************************/
//  FUNCTION:  rchisq():  returns a random chi-squared deviate.  It
//    merely calls rgamma() after checking the degrees of freedom.
//  Input:  double&
//  Return: double
//  Usage:  x = rchisq(nu)
//  Errors: 1011 
//  Dependencies:  math.h, rgamma(), rgamma1(), runif()
//    
//  7/10/2000 (ADM)

double rchisq (const double& nu);

/**********************************************************************/
//  FUNCTION:  rchisq():  returns a matrix of random chi-squared 
//    deviates.  See double rchisq() above for details.
//  Input:  int&, int&, double&
//  Return: Matrix
//  Usage:  X = rchisq(rows, cols, nu)
//  Errors: 1012 
//  Dependencies:  math.h, Scythe_Double_Matrix.h, rgamma(), 
//                rgamma1(), runif()
//    
//  7/10/2000 (ADM)

Matrix rchisq (const int& rows, const int& cols, const double& nu);

/**********************************************************************/
//  FUNCTION:  richisq():  returns a random inverse chi-squared deviate.
//    It merely calls the rigamma() function, after checking the 
//    degrees of freedom parameter.
//  Input:  double&
//  Return: double (pseudo-random inverse chi-squared deviate)
//  Usage:  x = richisq(nu)
//  Errors: 1013
//  Dependencies:  math.h, rgamma(), rgamma1(), runif(), rigamma()
//    
//  7/10/2000 (ADM)

double richisq (const double& nu);

/**********************************************************************/
//  FUNCTION:  richisq():  returns a matrix of random inverse chi-
//    squared deviates
//  Input:  int&, int&, double&
//  Return: Matrix (of pseudo-random inverse chi-squared deviates)
//  Usage:  X = richisq(rows, cols, nu)
//  Errors: 1014
//  Dependencies:  math.h, rgamma(), rgamma1(), runif(), rigamma()
//                Scythe_Double_Matrix.h
//    
//  7/10/2000 (ADM)

Matrix richisq (const int& rows, const int& cols, const double& nu);

/**********************************************************************/
//  FUNCTION:  rexp():  returns a random exponential deviate.  It calls
//    runif() and rescales based on the inverse shape parameter beta.
//  Input:  double&
//  Return: double (pseudo-random exponential deviate)
//  Usage:  x = rexp(beta)
//  Errors: 1015
//  Dependencies:  math.h, runif()
//    
//  7/10/2000 (ADM)

double rexp (const double& beta);

/**********************************************************************/
//  FUNCTION:  rexp():  returns a matrix o frandom exponential deviates.
//  Input:  int&, int&, double&
//  Return: Matrix (of pseudo-random exponential deviates)
//  Usage:  X = rexp(rows, cols, beta)
//  Errors: 1016
//  Dependencies:  math.h, Scythe_Double_Matrix.h, runif()
//    
//  7/10/2000 (ADM)

Matrix rexp (const int& rows, const int& cols, const double& beta);

/**********************************************************************/
//  FUNCTION:  rt():  returns a random Student t deviate.  It uses the
//    standard normal generator rnorm1() and the chisq generator
//    rchisq() to generate the draw.
//  Input:  double&, double&, double&
//  Return: double (pseudo-random student-t deviate)
//  Usage:  x = rt(mu,sigma2,nu)
//  Errors: 1017-1018
//  Dependencies:  math.h, rnorm1(), runif(), rchisq()
//    
//  7/10/2000 (ADM)

double rt (const double& mu = 0.0, const double& sigma2 = 1.0,
    const double& nu = 1.0);

/**********************************************************************/
//  FUNCTION:  rt():  returns a matrix of random Student t deviates.
//  Input:  int&, int&, double&, double&, double&
//  Return: Matrix (of pseudo-random student-t deviates)
//  Usage:  X = rt(rows, cols, mu, sigma2, nu)
//  Errors: 1019
//  Dependencies:  math.h, Scythe_Double_Matrix.h, rnorm1(), runif(), 
//                rchisq()
//    
//  7/10/2000 (ADM)

Matrix rt (const int& rows, const int& cols, const double& mu = 0.0,
    const double& sigma2 = 1.0, const double& nu = 1.0);

/**********************************************************************/
//  FUNCTION:  rbeta():  returns a random beta deviate, using the ratio
//    of chi-square rule.  It thus calls the rchisq() function twice.
//  Input:  double&, double&
//  Return: double (pseudo-random beta deviate)
//  Usage:  x = rbeta(alpha,beta)
//  Errors: 1020-1021
//  Dependencies:  math.h, rchisq()
//    
//  7/10/2000 (ADM)

double rbeta (const double& alpha, const double& beta);

/**********************************************************************/
//  FUNCTION:  rbeta():  returns a matrix  of random beta deviates,
//    using the ratio of chi-square rule.
//  Input:  int&, int&, double&, double&
//  Return: Matrix (of pseudo-random beta deviates)
//  Usage:  X = rbeta(rows,cols,alpha,beta)
//  Errors: 1022
//  Dependencies:  math.h, Scythe_Double_Matrix.h, rchisq()
//    
//  7/10/2000 (ADM)

Matrix rbeta (const int& rows, const int& cols, const double& alpha,
       const double& beta);

/**********************************************************************/
//  FUNCTION:  rlogis():  returns a random logistic deviate, using a
//    rescale of a unifom deviate.  Since the inverse CDF takes a
//    closed form, this is possible.  It calls the runif() function.
//  Input:  double&, double&
//  Return: double
//  Usage:  x = rlogis(alpha,beta)
//  Errors: 1023
//  Dependencies:  math.h, runif()
//    
//  7/10/2000 (ADM)

double rlogis (const double& alpha, const double& beta);

/**********************************************************************/
//  FUNCTION:  rlogis():  returns a matrix of random logistic deviates,
//    using a rescale of a unifom deviate. 
//  Input:  int&, int&, double&, double&
//  Return: Matrix
//  Usage:  X = rlogis(rows,cols,alpha,beta)
//  Errors: 1024
//  Dependencies:  math.h, Scythe_Double_Matrix.h, runif()
//    
//  7/10/2000 (ADM)

Matrix rlogis (const int& rows, const int& cols, const double& alpha, 
        const double& beta);

/**********************************************************************/
//  FUNCTION:  rbern():  returns a integer random Bernoulli deviate.
//    This calls the runif() function and accepts if it is less than p.
//  Input:  double&
//  Return: int  
//  Usage:  x = rbern(p)
//  Errors: 1025
//  Dependencies:  math.h, runif()
//    
//  7/10/2000 (ADM)

int rbern (const double& p);

/**********************************************************************/
//  FUNCTION:  rbern():  returns a matrix of integer random Bernoulli
//    deviates.
//  Input:  int&, int&, double&
//  Return: Matrix
//  Usage:  X = rbern(rows,cols,p)
//  Errors: 1026
//  Dependencies:  math.h, Scythe_Double_Matix.h, runif()
//    
//  7/10/2000 (ADM)

Matrix rbern (const int& rows, const int& cols, const double& p);

/**********************************************************************/
//  FUNCTION:  rbin():  returns a integer random binomial deviate.
//    This calls the runif() function n times, and counts the number of
//    successes.
//    Note:  for large n, there are more efficient ways to draw 
//      from this distribution.  If used extensively in an intensive 
//      program, it would be worth investigating 
//      these methods.
//  Input:  int&, double&
//  Return: int
//  Usage:  x = rbin(n,p)
//  Errors: 1027-1028
//  Depencies:  math.h, runif()
//    
//  7/10/2000 (ADM)

int rbin (const int& n, const double& p);

/**********************************************************************/
//  FUNCTION:  rbin():  returns a matrix of integer random binomial
//    deviates.
//  Input:  int&, int&, int&, double&
//  Return: Matrix
//  Usage:  X = rbin(rows,cols,n,p)
//  Errors: 1029
//  Depencies:  math.h, Scythe_Double_Matrix.h, runif()
//    
//  7/10/2000 (ADM)

Matrix rbin (const int& rows, const int& cols, const int& n,
             const double& p);

/**********************************************************************/
//  FUNCTION:  rwish():  returns a pseudo-random Wishart deviate w/
//    degrees of freedom v and scale matrix S
//  Input:  int&, Matrix&
//  Return: Matrix
//  Usage:  A = rwish(v, S)
//  Errors: 1030-1031
//  Depencies:  math.h, Scythe_Double_Matrix.h, rnorm()
//    
//  11/19/2000 (KQ)

Matrix rwish(const int& v, const Matrix& S);

/**********************************************************************/
//  FUNCTION:  gammln():  returns the log-Gamma function (which is
//    useful for computing factorials, and is used in various 
//    statistical functions).
//  Input:  double&
//  Return: double
//  Usage:  x = gammln(y)
//  Errors: 1032
//  Dependencies:  math.h
//
//
//  6/13/2000 (ADM)

double gammln (const double& x);

/**********************************************************************/
//  FUNCTION:  betaln():  returns the log-Beta function.
//  Input:  double&, double&
//  Return: double
//  Usage:  z = betaln(x, y)
//  Errors: none
//  Dependencies:  math.h
//
//  3/28/2001 (KQ)
double betaln(const double& x, const double& y);

/**********************************************************************/
//  FUNCTION:  beta():  returns the Beta function.
//  Input:  double&, double&
//  Return: double
//  Usage:  z = beta(x, y)
//  Errors: none
//  Dependencies:  math.h
//
//  3/28/2001 (KQ)
double beta (const double& x, const double& y);

/**********************************************************************/
//   FUNCTION:  rbetabin():  generates random deviates from the beta 
//    binomial distribution.  It does so by compounding rbeta() and 
//    rbin().
//  Input:  int&, double&, double&
//  Return: int
//  Usage:  x = rbetabin(n,alpha,beta)
//  Errors: 1033-1035
//  Dependencies:  math.h, rbeta(), rbin()
//    
//  7/10/2000 (ADM)

int rbetabin (const int& n, const double& alpha, const double& beta);

/**********************************************************************/
//   FUNCTION:  rbetabin():  generates random deviates from the beta 
//    binomial distribution.  It does so by compounding rbeta() and 
//    rbin().
//  Input:  int&, int&, int&, double&, double&
//  Return: Matrix
//  Usage:  X = rbetabin(rows,cols,n,alpha,beta)
//  Errors:  1036
//  Dependencies:  math.h, rbeta(), rbin(), Scythe_Double_Matrix.h
//    
//  7/10/2000 (ADM)

Matrix rbetabin (const int& rows, const int& cols, const int& n, 
                 const double& alpha, const double& beta);

/**********************************************************************/
//  FUNCTION: dnorm1():  returns the univariate standard normal density
//    evaluated at x
//  Input:  double& (point at which the USND is to be evaluated)
//  Return: double
//  Usage:  f = dnorm(x);
//  Errors: none
//  Dependencies:  none
//
//  6/10/2000 (KQ)

double dnorm1 (const double& x);

/**********************************************************************/
//  FUNCTION:  lndmvn():  returns the natural log of the multivariate
//    normal density with mean MU and covariance matrix SIGMA evaluated
//    at X.  (MU and X assumed to be column vectors).
//  Input:  Matrix&, Matrix&, Matrix&
//  Return: double
//  Usage:  f = lndmvn(X, MU, SIGMA);
//  Errors: 1037-1041
//  Dependencies:  none
//
// 11/19/2000 (KQ)

double lndmvn (const Matrix& x, const Matrix& mu, 
               const Matrix& Sigma);

/**********************************************************************/
//  FUNCTION:  pnorm1(x):  returns univariate standard normal cdf
//    evaluated at x
//  Input:  double
//  Return: double
//  Usage:  f = pnorm(x);
//  Errors: 1042-1043
//  Dependencies:  none
//  References:
//    This algorithm is based on the power series expansion given in:
//    Lange, Kenneth. 1999. Numerical Analysis for Statisticians. 
//    Section 2.4
//
//    with default settings this is accurate for 1e-10 < x < 1 - 1e10
//
//  6/10/2000 (KQ)

double pnorm1 (const double& xx, const double& eps=1.0e-50, 
	       const double& maxit=500, const double& minit=10);

/**********************************************************************/
//  FUNCTION:  qnorm1(p):  returns the quantile of the standard normal
//    distribution associated with a given probability p
//  Input:  double&
//  Return: double
//  Usage:  q = qnorm1(p);
//  Errors: 1044
//  Dependencies:  none
//  References:
//    This code is based on the GAUINV Algorithm given in Kennedy and 
//    Gentle, 1980, _Statistical Computing_, p. 95.
//
//   this code has 7 digit accuracy for  10e-20 < p < 0.5
//
// 8/3/2000 (KQ)

double qnorm1 (const double& in_p);

/**********************************************************************/
//  FUNCTION:  dt1():  returns the univariate student-t density
//      evaluated at x with mean mu, scale sigma2, and dof nu
//  Input:  double&, double&, double&, double&
//  Return: double
//  Usage:  f = dt1(x,mu,sigma2,nu);
//  Errors: none
//  Dependencies:  none
//
//  3/25/2001 (KQ)

double dt1(const double& x, const double& mu, const double& sigma2, 
           const double& nu);  

/**********************************************************************/
//  FUNCTION:  logdt1():  returns the natural log of the univariate
//    student-t density evaluated at x with mean mu, scale sigma2, and
//    dof nu
//  Input:  double&, double&, double&, double&
//  Return: double
//  Usage:  f = logdt1(x,mu,sigma2,nu);
//  Errors: none
//  Dependencies:  none
//
//  3/25/2001 (KQ)

double  logdt1(const double& x, const double& mu, const double& sigma2, 
               const double& nu);

/**********************************************************************/
//  FUNCTION:  logdbeta1():  returns the natural log of the ordinate of
//    the beta density evaluated at x, with shape1 a and shape2 b
//  Input:  double&, double&, double&
//  Return: double
//  Usage:  f = logdbeta1(x,a,b);
//  Errors: 1045-1047
//  Dependencies:  none
//
//  3/28/2001 (KQ)

double logdbeta1(const double& x, const double& a, const double& b);

/**********************************************************************/
//  FUNCTION:  dbeta1:  returns the univariate beta density evaluated
//    at x with shape1 a and shape2 b
//  Input:  double&, double& double&
//  Return: double
//  Usage:  f = dbeta1(x,a,b);
//  Errors: 1048-1050
//  Dependencies:  none
//
//  3/28/2001 (KQ)

double dbeta1(const double& x, const double& a, const double& b);

/**********************************************************************/
//  FUNCTION:  rtnorm():  Simulate from a truncated normal distribution
//    with mean parameter m, variance parameter v on the interval
//    (below,above). Uses the inverse cdf method.
//  Input:  double&, double&, double&, double&
//  Return: double
//  Usage:  x = rtnorm(m,v,below,above);
//  Errors: 1051, XXXX
//  Dependencies:  none

double rtnorm(const double& m, const double& v, const double& below, 
              const double& above);  


/**********************************************************************/
//  FUNCTION: rtbnorm_slice() Sample from a normal distribution with
//    mean m, variance v, truncated to the interval (below, infinity)
//    via the slice sampler of Robert and Casella (1999), pp. 288-289. 
//    Sampler is run for iter iterations.
//    iter takes a default argument of 10.
//    @@@ NB: THIS METHOD ONLY WORKS WHEN below >= m @@@ 
//  Input:  const double&, const double&, const double&, const int&
//  Return: double
//  Usage:  x = rtbnorm_slice(m,v,below);
//          x = rtbnorm_slice(m,v,below,200);
//  Errors: XXXX
//  Dependencies: none
double rtbnorm_slice(const double& m, const double& v, const double& below, 
		     const int& iter=10);

/**********************************************************************/
//  FUNCTION: rtanorm_slice() Sample from a normal distribution with
//    mean m, variance v, truncated to the interval (-infinity, above)
//    via the slice sampler of Robert and Casella (1999), pp. 288-289. 
//    Sampler is run for iter iterations.
//    iter takes a default argument of 10.
//    @@@ NB: THIS METHOD ONLY WORKS WHEN above <= m @@@ 
//  Input:  const double&, const double&, const double&, const int&
//  Return: double
//  Usage:  x = rtanorm_slice(m,v,above);
//          x = rtanorm_slice(m,v,above,200);
//  Errors: XXXX
//  Dependencies: none
double rtanorm_slice(const double& m, const double& v, const double& above, 
		     const int& iter=10);


/**********************************************************************/
//  FUNCTION: rtbnorm_combo() Sample from a normal distribution with
//    mean m, variance v, truncated to the interval (below, infinity)
//    using a combination of rejection sampling (when m >= below) and 
//    the slice sampling method of Robert and Casella (1999), pp. 
//    288-289 (when m < below).  
//    Slice sampler is run for iter iterations.
//    iter takes a default argument of 10.
//  Input:  const double&, const double&, const double&, const int&
//  Return: double
//  Usage:  x = rtbnorm_combo(m,v,below);
//          x = rtbnorm_combo(m,v,below,200);
//  Errors: XXXX
//  Dependencies: none
double rtbnorm_combo(const double& m, const double& v, const double& below,
		     const int& iter=10);


/**********************************************************************/
//  FUNCTION: rtanorm_combo() Sample from a normal distribution with
//    mean m, variance v, truncated to the interval (-infinity, above)
//    using a combination of rejection sampling (when m <= above) and 
//    the slice sampling method of Robert and Casella (1999), pp. 
//    288-289 (when m > above).  
//    Slice sampler is run for iter iterations.
//    iter takes a default argument of 10.
//  Input:  const double&, const double&, const double&, const int&
//  Return: double
//  Usage:  x = rtanorm_combo(m,v,above);
//          x = rtanorm_combo(m,v,above,200);
//  Errors: XXXX
//  Dependencies: none
double rtanorm_combo(const double& m, const double& v, const double& above,
		     const int& iter=10);



/**********************************************************************/
//  FUNCTION:  factorial():  Computes the factorial !n of a non-negative
//    integer
//  Input:  int
//  Return: int
//  Usage:  x = factorial(n)
//  Errors: 1052
//  Dependencies:  none
int factorial (int n);


/**********************************************************************/
//  FUNCTION:  logfact():  Computes the log factorial !n of a non-negative
//    integer
//  Input:  int
//  Return: int
//  Usage:  y = logfact(n)
//  Errors:
//  Dependencies:  none
double logfact(const int& n);


/**********************************************************************/
//  FUNCTION:  dpois(): Computes the Poisson probability mass function
//  Input:     const int&, const double&
//  Return:    double
//  Usage:     double d = dpois(x, lambda)
//  Errors: ????
//  Dependencies:   ????
double dpois(const int& x, const double& lambda);

/**********************************************************************/
//  FUNCTION:  logdpois(): Computes the log of the Poisson probability 
//                         mass function
//  Input:     const int&, const double&
//  Return:    double
//  Usage:     double d = logdpois(x, lambda)
//  Errors: ????
//  Dependencies:   ????
double logdpois(const int& x, const double& lambda);


/**********************************************************************/
//  FUNCTION:  rpois(): generates a Poisson pseudo-random deviate
//  Input:     const double&
//  Return:    int
//  Usage:     int r = rpois(lambda)
//  Errors: ????
//  Dependencies: ????
int rpois(const double& lambda);


/**********************************************************************/
//  FUNCTION:  bubbleSort():  sorts a column vector (slowly) and returns
//    the sorted vector
//  Input:  Matrix&
//  Return: Matrix
//  Usage:  S = bubbleSort(U);
//  Errors: 1053
//  Dependencies

Matrix bubbleSort (const Matrix& A);

} // namespace dec

#endif /* SCYTHE_SIMULATE_H */
