///////////////////////////////////////////////////////////////////////////
//
// irt1ddlm.cc fits a longitudinal irt model to the court data 1953-1999
//
// the model is:
//
//   z_jk = alpha_k + beta_k * theta_jt + epsilon_jk
//
//   alpha_k ~ N(0, Sigma_alpha)
//   beta_k  ~ N(0, Sigma_beta)
//   theta_jt ~ N(theta_j(t-1), Sigma_theta)
//
//   justice j's ideal point (theta_j) is assumed to be constant within a 
//     term, but to follow a first order DLM (West and Harrison, 1997), 
//     i.e. a random walk, across terms. 
//
//
/////////////////////////////////////////////////////////////////////////// 
// 
// Kevin Quinn
// Dept. of Political Science and CSSS
// University of Washington
// Seattle, Washington
// quinn@stat.washington.edu
//
// Andrew D. Martin
// Department of Political Science
// Washington University
// St. Louis, Missouri
// admartin@artsci.wustl.edu
//
///////////////////////////////////////////////////////////////////////////
//
// This program is another Quinn Martin production. Copyright (C) 2001.
//
/////////////////////////////////////////////////////////////////////////// 
//
// INPUT FILES:
//
//    1953.txt - 1999.txt: (K[t] \times J[t]) matrices of votes for 
//               each term
//    adj.txt: (T \times J_total) adjacency matrix to match justices 
//             across terms
//    case.txt: (K_total \times Q_case) matrix of case-specific covariates
//    justice.txt: (J_total \times Q_justice) matrix of justice-specific 
//                 covariates
//    term.txt: (T \times Q_term) matrix of term-specific covariates
//
// UPDATED: 6/14/2001 (KMQ)
//
/////////////////////////////////////////////////////////////////////////// 
 

#include <iostream.h>
#include "/ws/home/quinn/Scythe0.2/Scythe_Double_Matrix.h"
#include "/ws/home/quinn/Scythe0.2/Scythe_Simulate.h" 
#include "/ws/home/quinn/Scythe0.2/Scythe_Optimize.h"

using namespace SCYTHE;

#include <string>

// convert integers to strings
#include <stdio.h>
string itostring( long int i ) {
  char buffer[20];
  sprintf( buffer, "%ld", i );
  return string( buffer );
}





int main() { 

   //
   // Global Parameters
   //
   int T = 47;         // terms
   int J_total = 29;   // total number of justices
   int K_total = 3450; // total number of cases
   int Q_case = 53;    // number of case-specific covariates
   int Q_justice = 3;  // number of justice-specific covariates
   int Q_term = 6;     // number of term-specific covariates

   //
   // Justice Parameters
   //
   int J[T]; // justices serving in each term
   for(int j=0; j<T; j++) J[j] = 9;
   J[3] = 10;  // ten justices during 1956 term (Reed off, Brennan on mid-term)
   J[8] = 10;  // ten justices during 1961 term (Whittaker off, Stewart on mid-term)
   J[16] = 8;  // ten justices during 1969 term (Fortas not replaced until 1970)
   
   cout << "Justices in Each Term" << endl;
   for(int term=0; term<T; term++)
      cout << term + 1953 << " " << J[term] << endl;
    
   //   
   // Case Parameters
   //
   int K[T]; // cases heard in each term
   
   K[0]  = 52;  K[1] = 50;  K[2] = 55;  K[3] = 86;  K[4] = 90;  K[5] = 67;
   K[6]  = 83;  K[7] = 85;  K[8] = 64;  K[9] = 76; K[10] = 72; K[11] = 60;
   K[12] = 58; K[13] = 75; K[14] = 75; K[15] = 68; K[16] = 68; K[17] = 78;
   K[18] = 89; K[19] =102; K[20] = 99; K[21] = 84; K[22] = 88; K[23] = 92;
   K[24] = 86; K[25] = 81; K[26] = 96; K[27] = 80; K[28] = 92; K[29] = 90;
   K[30] = 83; K[31] = 77; K[32] = 100; K[33] = 104; K[34] = 77; K[35] = 82;
   K[36] = 86; K[37] = 69; K[38] = 63; K[39] = 57; K[40] = 49; K[41] = 47;
   K[42] = 42; K[43] = 40; K[44] = 43; K[45] = 46; K[46] = 44;        

   cout << endl << "Total Cases: " << K_total << endl << endl;
   cout << "Cases in Each Term" << endl;
   for(int term=0; term<T; term++)
      cout << term + 1953 << " " << K[term] << endl;

   //
   // Load Vote Matrices
   //
   // Note: Each vote Matrix is (K[t] \times J[t]). Each vote is
   // coded 1 = affirm, 0 = reverse, and 9 = missing.
   //
   Matrix Y_big[T]; // vote Matrix for each term
   string s1 = "/ws/home/quinn/pafinal/new_data/no_unan/vote/";
   string s3 = ".txt";
   
   for(int term=0; term<T; ++term) {
      string s2 = itostring(term + 1953);
      string file = s1 + s2 + s3;
      const char *cfile = file.c_str();
      Y_big[term] = t(read_into_matrix(cfile, K[term], J[term]));
      }

   //      
   // Load Adjacency Matrix
   //
   // Note: This is a (T \timtes J_total) Matrix which allows us to match
   // justices across terms.  It takes an entry of the row number (0,...,J[t])
   // if the justice served in term t, and -9 if the justice did not serve in
   // the term.
   //
   // The order of the justices in this Matrix as well as in the justice-
   // specific covariates is (the name abbreviation is followed by column number):
   //
   // HAR 0, BLC 1, DOUG 2, STWT 3, MAR 4, BRN 5, BW 6, WAR 7, CLK 8, FRK 9,
   // WHIT 10, BURT 11, REED 12, FORT 13, GOLD 14, MINT 15, JACK 16, BURG 17,
   // BLKM 18, POW 19, REHN 20, STEV 21, OCON 22, SCAL 23, KEN 24, SOUT 25,
   // THOM 26, GIN 27, BRY 28
   //
   Matrix adjacency;
   adjacency = read_into_matrix("/ws/home/quinn/pafinal/new_data/adj.txt",T,J_total);

   //           
   // Load Case-Specific Covariates
   //
   // Note: This is a (K_total \times Q_case) Matrix of case-specific covariates.
   // The covariates are:
   //
   // criminal procedure dummy 0, civil rights dummy 1, first amendment dummy 2,
   // due process dummy 3, privacy dummy 4, attorney and union dummy 5,
   // economics dummy 6, judicial power dummy 7, federalism dummy 8, 
   // 1st curcuit dummy 9 - 11th circuit dummy 19, DC circuit dummy 20, 
   // constitutional case dummy 21, declaration of unconstitutionality dummy 22,
   // formal alteration of precedent dummy 23, harlan opinion writer dummy 24 -
   // breyer opinion writing dummy 52 
   //
   Matrix case_specific;
   case_specific = read_into_matrix("/ws/home/quinn/pafinal/new_data/no_unan/case/case.txt", K_total, Q_case);


   for (int i=24; i<53; ++i){
     Matrix op_write = case_specific(_, i);
   }
   //    
   // Load Justice-Specific Covariates
   //
   // Note: This is a (J_total \times Q_justice) Matrix of justice-specific
   // covariates.  The covariates are:
   //
   // party of appointing president [0=GOP, 1=Dem] 0, year of appointment 1,
   // Segal / Cover score 2
   // 
   Matrix justice_specific;
   justice_specific = read_into_matrix("/ws/home/quinn/pafinal/new_data/justice.txt", J_total, Q_justice); 

   
   //
   // Load Term-Specific Covariates 
   //
   // Note: This is a (T \times Q_term) Matrix of term-specific covariates. 
   // The covariates are:
   //
   // party of president [0=GOP, 1=Dem] 0, party of Senate 1,
   // Senate seat differential [Dem - GOP] 2, party of House 3, House
   // seat differential 4, unified government variable 5
   //
   Matrix term_specific = Matrix(T,Q_term);
   term_specific = read_into_matrix("/ws/home/quinn/pafinal/new_data/term.txt", T, Q_term);   



   // model parameters
   int D = 1; // dimension of the issue space


   // create lag_theta_index which is an array of matrices (same 
   // dimensions as theta_big). lag_theta_index[term][j] gives the
   // index of theta_big[term] that corresponds to justice j's 
   // ideal point in term t-1. If lag_theta_index[term][j] == -9
   // then there is no ideal point for justice j in term t-1.
   //
   // NOTE: currently only setup to work for 1D models
   Matrix lag_theta_index[T];
   for (int term=0; term<T; ++term){
     if (term==0)
       lag_theta_index[term] = -9*ones(J[term], D);
     else {
       lag_theta_index[term] = -9*ones(J[term], D);
       for (int j=0; j<J_total; ++j){
	 int cur_ind = static_cast<int>(adjacency(term,j));
	 if (cur_ind >= 0){
	   lag_theta_index[term][cur_ind] = adjacency(term-1,j);
	 }
       }
     }
   }
   
   // create forlag_theta_index which is an array of matrices (same 
   // dimensions as theta_big). forlag_theta_index[term][j] gives the
   // index of theta_big[term] that corresponds to justice j's 
   // ideal point in term t+1. If forlag_theta_index[term][j] == -9
   // then there is no ideal point for justice j in term t+1.
   //
   // NOTE: currently only setup to work for 1D models
   Matrix forlag_theta_index[T];
   for (int term=0; term<T; ++term){
     if (term==(T-1))
       forlag_theta_index[term] = -9*ones(J[term], D);
     else {
       forlag_theta_index[term] = -9*ones(J[term], D);
       for (int j=0; j<J_total; ++j){
	 int cur_ind = static_cast<int>(adjacency(term,j));
	 if (cur_ind >= 0){
	   forlag_theta_index[term][cur_ind] = adjacency(term+1,j);
	 }
       }
     }
     forlag_theta_index[term].print();
     cout << endl << endl;
   }


   // create terms_per_just which is an array of ints that holds the
   // number of terms each justice served. Ordering of the array is
   // the same as that in adjacency; i.e., HAR 0, BLC 1, DOUG 2, STWT
   // 3, MAR 4, BRN 5, BW 6, WAR 7, CLK 8, FRK 9, WHIT 10, BURT 11,
   // REED 12, FORT 13, GOLD 14, MINT 15, JACK 16, BURG 17, BLKM 18,
   // POW 19, REHN 20, STEV 21, OCON 22, SCAL 23, KEN 24, SOUT 25,
   // THOM 26, GIN 27, BRY 28
   // This also creates first_term_just which is an array of ints 
   // giving the term in which justice j first served
   int terms_per_just[J_total];
   int first_term_just[J_total];
   for (int j=0; j<J_total; ++j){
     terms_per_just[j] = 0;
     first_term_just[j] = -999;
     for (int term=0; term<T; ++term){
       if(adjacency(term,j) >= 0){
	 ++terms_per_just[j];
	 if (first_term_just[j] < 0)
	   first_term_just[j] = term;
       }
     }
     cout << "justice # = " << j << "  terms in career = " 
	  << terms_per_just[j] << "  first term = " << 
       first_term_just[j] <<  endl;
   }
   


  // MCMC parameters
  int burnin = 20000;
  int gibbs  = 200000;
  int thin = 100; 
  int tot_iter = burnin + gibbs;  

  // prior on \eta_j
  // \eta_j' = (\alpha_j \beta_j') ~ N(b0, B0^-1)
  Matrix b0 = Matrix(D+1,1);
  Matrix B0 = eye(D+1)*1; // precision Matrix (NOT covariance)
  Matrix B0b0 = B0*b0;

  // priors for theta_0 (defined to be justice specific)
  // t0 mean 
  // T0 variance
  Matrix t0[J_total];
  Matrix T0[J_total];
  for (int j=0; j<J_total; ++j){ 
    t0[j] = Matrix(D,1);
    T0[j] = eye(D);
  }
  // Harlan
  t0[0] = ones(D,1)*1;
  T0[0] = eye(D)*.1;
  // Douglas
  t0[2] = ones(D,1)*-3;
  T0[2] = eye(D)*.1;
  // Marshall
  t0[4] = ones(D,1)*-2;
  T0[4] = eye(D)*.1;
  // Brennan
  t0[5] = ones(D,1)*-2;
  T0[5] = eye(D)*.1;
  // Franklin
  t0[9] = ones(D,1)*1;
  T0[9] = eye(D)*.1;
  // Fortas
  t0[13] = ones(D,1)*-1;
  T0[13] = eye(D)*.1;
  // Rehnquist
  t0[20] = ones(D,1)*2;
  T0[20] = eye(D)*.1;
  // Scalia
  t0[23] = ones(D,1)*2.5;
  T0[23] = eye(D)*.1;
  // Thomas
  t0[26] = ones(D,1)*2.5;
  T0[26] = eye(D)*.1;


  // prior variance for theta_i in all terms after the first
  Matrix T1all = eye(D)*.1;
  Matrix T1Doug = eye(D)*.001;

  // storage matrices (row major order)
  //Matrix eta_store   = Matrix(gibbs,K*(D+1));
  Matrix theta_store[T];
  Matrix avg_correct[T];

  // starting values
  Matrix Z_big[T]; 
  Matrix theta_big[T];
  Matrix alpha_big[T];
  Matrix beta_big[T]; 

  for (int i=0; i<T; ++i){
    theta_store[i] = Matrix(gibbs/thin, J[i]*D);
    avg_correct[i] = Matrix(J[i], K[i]);
    Z_big[i] = rnorm(J[i], K[i]);
    theta_big[i] = rnorm(J[i], D);
    alpha_big[i] = rnorm(K[i], 1);
    beta_big[i] = rnorm(K[i], 1);
  }



  ///////////////////
  // Gibbs Sampler //
  ///////////////////
  int count = 0;
  for (int iter=0; iter < tot_iter; ++iter){

    // sample theta via forward filtering backward sampling
    // notation is basically that of West and Harrison 1997
    for (int j=0; j<J_total; ++j){
      // create quantities needed for the Kalman filter
      // all quantities for case of unidimensional theta 
      Matrix T1;
      if (j==2) // put stronger prior on Douglas
	T1 = T1Doug;
      else
	T1 = T1all;
      int t_j = terms_per_just[j];
      int first_term = first_term_just[j];
      Matrix a[t_j];
      Matrix R[t_j];
      Matrix f[t_j];
      Matrix Q[t_j];
      Matrix e[t_j];
      Matrix A[t_j];
      Matrix m[t_j];
      Matrix C[t_j];
      // run the filter forward through time
      int j_row = static_cast<int>(adjacency(first_term,j));
      Matrix Z = Z_big[first_term];
      Matrix Z_j = t(Z(j_row,_));
      a[0] = t0[j];
      R[0] = T0[j] + T1;
      f[0] = beta_big[first_term] * a[0];
      Q[0] = beta_big[first_term] * R[0] * t(beta_big[first_term]) + 
	eye(K[first_term]);
      e[0] = Z_j - alpha_big[first_term] - f[0];
      A[0] = R[0] * t(beta_big[first_term]) * invpd(Q[0]);
      m[0] = a[0] + A[0]*e[0];
      C[0] = R[0] - A[0]*Q[0]*t(A[0]);
      int theta_count = 1;

      for (int term=first_term+1; term<(t_j+first_term); ++term){
	j_row = static_cast<int>(adjacency(term,j));
	Z = Z_big[term];
	Z_j = t(Z(j_row,_));
	a[theta_count] = m[theta_count-1];
	R[theta_count] = C[theta_count-1] + T1;
	f[theta_count] = beta_big[term] * a[theta_count];
	Q[theta_count] = beta_big[term] * R[theta_count] * t(beta_big[term]) + 
	  eye(K[term]);
	e[theta_count] = Z_j - alpha_big[term] - f[theta_count];
	A[theta_count] = R[theta_count] * t(beta_big[term]) * 
	  invpd(Q[theta_count]);
	m[theta_count] = a[theta_count] + A[theta_count]*e[theta_count];
	C[theta_count] = R[theta_count] - A[theta_count] * 
	  Q[theta_count]*t(A[theta_count]);	
	++theta_count;
      }
      
      
      // sample theta for the last term j served
      j_row = static_cast<int>(adjacency(first_term+t_j-1,j));
      theta_big[first_term+t_j-1][j_row] = rnorm(m[t_j-1][0], C[t_j-1][0]); 
      if (iter < 20){
	if (j==2)
	  theta_big[first_term+t_j-1][j_row] = -3;
	else if (j==4 || j==5)
	  theta_big[first_term+t_j-1][j_row] = -2;
	else if (j==20)
	  theta_big[first_term+t_j-1][j_row] = 2;
	else if (j==23)
	  theta_big[first_term+t_j-1][j_row] = 2.5;
	else if (j==26)
	  theta_big[first_term+t_j-1][j_row] = 2.5;
      }
      
      
      
      // sample remaining thetas backward through time
      theta_count = t_j-2;
      for (int term=(first_term+t_j-2); term>=first_term; --term){
	j_row = static_cast<int>(adjacency(term,j));
	int j_rowp1 = static_cast<int>(adjacency(term+1,j));
	Matrix B = C[theta_count] * 1 * invpd(R[theta_count+1]);
	Matrix h = m[theta_count] + B * (theta_big[term+1][j_rowp1] - 
				  a[theta_count+1]);
	Matrix H = C[theta_count] - B * R[theta_count+1] * t(B);
	theta_big[term][j_row] = rnorm(h[0], H[0]);
	if (iter < 20){
	  if (j==2)
	    theta_big[term][j_row] = -3;
	  else if (j==4 || j==5)
	    theta_big[term][j_row] = -2;
	  else if (j==20)
	    theta_big[term][j_row] = 2;
	  else if (j==23)
	    theta_big[term][j_row] = 2.5;
	  else if (j==26)
	    theta_big[term][j_row] = 2.5;
	}
	--theta_count;
      }
    }
    
 
    
    for (int term=0; term<T; ++term){
      // get the term specific data and parameters
      Matrix Y = Y_big[term];
      Matrix Z = Z_big[term];
      Matrix theta = theta_big[term];
      Matrix alpha = alpha_big[term];
      Matrix beta = beta_big[term];
      
      
      // sample Z
      for (int j=0; j<J[term]; ++j){
	for (int k=0; k<K[term]; ++k){
	  double Z_mean = alpha[k] + beta[k]*theta[j];
	  if (Y(j,k) == 1.0){
	    Z(j,k) = rtbnorm_combo(Z_mean, 1.0, 0);
	    if(isinf(Z(j,k)))
	      Z(j,k) = 0;
	  }
	  if (Y(j,k) == 0.0){
	    Z(j,k) = rtanorm_combo(Z_mean, 1.0, 0);
	    if(isinf(Z(j,k)))
	      Z(j,k) = 0;
	  }
	  if (Y(j,k) != 1.0 && Y(j,k) != 0.0)
	    Z(j,k) = rnorm(Z_mean, 1.0);
	}
      }
      
        
      // sample eta
      Matrix theta_star = cbind(ones(J[term],1),theta);
      Matrix post_var = invpd(crossprod(theta_star) + B0);
      Matrix Cpost_var = cholesky(post_var);
      for (int k=0; k<K[term]; ++k){
	Matrix post_mean = post_var * gaxpy(t(theta_star), 
					    Z(_,k), B0b0);
	Matrix new_eta =  gaxpy(Cpost_var,  rnorm(D+1,1), post_mean);
	alpha[k] = new_eta[0]; // needs to be changed is D>1 
	beta[k] = new_eta[1];  // needs to be changed is D>1
	//for(int d = 0; d<D+1; ++d)
	//	eta(k,d) = new_eta[d];
      }
      
     
      
      
      // if after burn-in compute predicted values and save
      Matrix ypredhold = Matrix(J[term],K[term]);
      for (int j=0; j<J[term]; ++j){
	for (int k=0; k<K[term]; ++k){   
	  double Z_mean = alpha[k] + beta[k]*theta[j];       
	  double Zpred = rnorm(Z_mean, 1.0);
	  ypredhold(j,k) = (Zpred>0);
	}
      }
      
      // initialize for first iteration
      if (iter==burnin){
	avg_correct[term] = (ypredhold^=Y);
      }
      else{
	Matrix avg_correct_t = (ypredhold^=Y);
	double itcount = static_cast<double>(iter-burnin);
	avg_correct[term] = (1.0/(1.0+itcount))* 
	  (itcount*avg_correct[term] + avg_correct_t);
      }

      
      
      
      // print results to screen
      if ( (iter % 5) == 0){
	cout << "Gibbs Iteration = " << iter << endl;
	t(theta).print();      
      }
      
      // store results
      if ((iter >= burnin) && ((iter % thin)==0) ){
	Matrix theta_store_vec = reshape(theta,1,J[term]*D);
	for (int l=0; l<J[term]*D; ++l)
	  theta_store[term](count, l) = theta_store_vec[l];
	if (term==(T-1))
	  ++count;
      }

      // put the term specific data and parameters back
      Y_big[term] = Y;
      Z_big[term] = Z;
      theta_big[term] = theta;
      alpha_big[term] = alpha;
      beta_big[term] = beta;
    }// end term loop  
  } // end Gibbs loop
  
  
  ////////////////////////
  // Write Output Files //
  ////////////////////////
  Matrix career_path = Matrix(T, J_total);
  Matrix career_std  = Matrix(T, J_total);

  for (int term=0; term<T; ++term){
    Matrix theta_mean = meanc(theta_store[term]);
    Matrix theta_std  = stdc(theta_store[term]);
    for (int j=0; j<J_total; ++j){
      int colj = static_cast<int>(adjacency(term,j));
      if (colj >= 0){
	career_path(term, j) = theta_mean[colj];
	career_std(term, j) = theta_std[colj];
      }
    }

    Matrix theta_out = cbind(t(meanc(theta_store[term])), 
			     t(stdc(theta_store[term])));
    
    cout << endl;
    cout << "@@@@@@@@@@@@@@@@@@@@@@@" << endl;
    cout << "Theta Posterior Summary" << endl;
    cout << "term " << term << endl;
    cout << "@@@@@@@@@@@@@@@@@@@@@@@" << endl;
    cout << endl;
    cout << "  PostMean PostStdDev" << endl;
    theta_out.print();
    
    cout << "avg. correct" << endl;
    meanc(t(avg_correct[term])).print();



   string s1a = "theta";
   string s1b = "avgcorrect";
   string s3 = ".txt";
   
   string s2 = itostring(term);
   string filea = s1a + s2 + s3;
   string fileb = s1b + s2 + s3;
   const char *cfilea = filea.c_str();
   const char *cfileb = fileb.c_str();
    
   // write_Matrix_out (eta_store, "eta.txt", "o", false);
   write_matrix_out (theta_store[term], cfilea, "o", false);
   write_matrix_out(t(avg_correct[term]), cfileb, "o", false);
  }

  write_matrix_out(career_path, "careermean.txt", "o", false);
  write_matrix_out(career_std, "careerstd.txt", "o", false);
  

} // end main()














